<?php

namespace App\Http\Controllers\pages;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;
use App\Services\DSpaceApiService;
use GuzzleHttp\Exception\ClientException;
use Illuminate\Support\Facades\Log;

class CollectionController extends Controller
{
    protected $title;

    private function getTitleFieldId()
    {
        return DB::table('metadatafieldregistry')
            ->where('element', 'title')
            ->whereNull('qualifier')
            ->where('metadata_schema_id', 1)
            ->value('metadata_field_id');
    }

    //index funcation for collection
    public function index($id)
    {
        $title = $this->getTitleFieldId();
        $community_name = DB::table('metadatavalue')
            ->where('dspace_object_id', $id)
            ->where('metadata_field_id', $title)
            ->value('text_value'); // Fetches a single column value

        // Fetch data from the database for collection
        $collectionData = DB::table('metadatavalue as md')
            ->select(
                'md.dspace_object_id',
                'md.text_value as collection_name',
                'md2.text_value as entity_type',
                'h.handle'
            )
            ->join('public.collection as c', 'md.dspace_object_id', '=', 'c.uuid')
            ->join('community2collection as cc', 'md.dspace_object_id', '=', 'cc.collection_id')
            ->join('metadatavalue as md2', function ($join) {
                $join->on('md2.dspace_object_id', '=', 'cc.collection_id')
                    ->where('md2.metadata_field_id', '=', 7);
            })
            ->join('public.handle as h', 'h.resource_id', '=', 'cc.collection_id')
            ->where('cc.community_id', $id) // Use the passed ID here
            ->where('md.metadata_field_id', $title)
            ->groupBy('md.dspace_object_id', 'md.text_value', 'md2.text_value', 'h.handle_id', 'h.handle')
            ->get();

        // Count the results of collection
        $count = $collectionData->count();

        // Fetch data for sub-community and check if it exists
        $results = DB::table('metadatavalue')
            ->select('text_value', 'dspace_object_id')
            ->whereIn('dspace_object_id', function ($query) use ($id) {
                $query->select('child_comm_id')
                    ->from('community2community')
                    ->where('parent_comm_id', $id);
            })
            ->where('metadata_field_id', $title)
            ->get();
        // Check if results exist
        $hasSubCommunity = $results->isNotEmpty() ? 'yes' : 'no';
        $subCommunityData = $hasSubCommunity === 'yes' ? $results : null;

        // Pass the data and count to the view
        return view('content.pages.collection', compact('collectionData', 'count', 'id', 'community_name', 'hasSubCommunity', 'subCommunityData'));
    }

    // Add collection Function
    public function addCollection(Request $request, DSpaceApiService $apiService)
    {
        $id = $request->input('id'); // Retrieve ID from the request

        // Prepare the data as per the API format
        $data = [
            'name' => $request->name,
            'metadata' => [
                'dc.title' => [
                    [
                        'value' => $request->name,
                        'language' => null,
                        'authority' => null,
                        'confidence' => -1
                    ]
                ],
                'cris.submission.definition' => [
                    [
                        'value' => $request->entity_action,
                        'language' => null,
                        'authority' => null,
                        'confidence' => -1,
                        'place' => 0
                    ]
                ],
                'dspace.entity.type' => [
                    [
                        'value' => $request->entity_type,
                        'language' => null,
                        'authority' => null,
                        'confidence' => -1,
                        'place' => 0
                    ]
                ]
            ]
        ];

        // Fetch the API token from the session
        $authorizationHeader = session()->get('api_token');  // Assuming the token is saved in the session

        if (!$authorizationHeader) {
            return back()->with('error', 'Authorization header not found.');
        }
        try {
            $response = $apiService->post('core/collections?parent=' . $id, $data, $authorizationHeader);
            // Check if the response was successful
            if ($response->getStatusCode() === 201) {
                return back()->with('success', 'Collection created successfully!');
            } else {
                $errorBody = (string) $response->getBody();
                return back()->with('error', 'Failed to create community. Error: ' . $errorBody);
            }
        } catch (\Exception $e) {
            // Handle any exceptions (network issues, etc.)
            return back()->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }

    // Delete Collection Function
    public function deleteCollection($id, DSpaceApiService $apiService)
    {
        $authorizationHeader = Session::get('api_token');

        $response = $apiService->delete('core/collections/' . $id, $authorizationHeader);


        // Check if the response was successful
        if ($response->getStatusCode() === 204) {
            // Redirect to the desired view with a success message
            return redirect()->back()->with('success', 'Collection deleted successfully');
        } else {
            // Redirect back with an error message or handle the error accordingly
            return redirect()->back()->with('error', 'Failed to delete process');
        }
    }

    // Edit Collection Function
    public function editCollection(Request $request)
    {
        // Get ID and new name from the request
        $id = $request->input('id');
        $name = $request->input('collection_name');
        $title = $this->getTitleFieldId();
        // Update the database
        DB::table('metadatavalue as md')
            ->join('collection as c', 'md.dspace_object_id', '=', 'c.uuid')
            ->where('md.metadata_field_id', $title)
            ->where('md.dspace_object_id', $id)
            ->update(['md.text_value' => $name]);


        // Redirect back with success message
        return redirect()->back()->with('success', 'Collection updated successfully!');
    }


    // Sub Community Functions start from her

    // Add Sub Community Function
    public function addSubCommunity(Request $request, DSpaceApiService $apiService)
    {
        $id = $request->input('id'); // Retrieve parent community ID from the request

        // Prepare the data as per the API format
        $data = [
            'name' => $request->name,
            'metadata' => [
                'dc.title' => [
                    [
                        'value' => $request->name,
                        'language' => null,
                        'authority' => null,
                        'confidence' => -1
                    ]
                ]
            ]
        ];

        // Fetch the API token from the session
        $authorizationHeader = session()->get('api_token');  // Assuming the token is saved in the session

        if (!$authorizationHeader) {
            return back()->with('error', 'Authorization header not found.');
        }
        try {
            $response = $apiService->post('core/communities?parent=' . $id, $data, $authorizationHeader);

            if ($response->getStatusCode() === 201) {
                return back()->with('success', 'Community created successfully!');
            } else {
                $errorBody = (string) $response->getBody();
                return back()->with('error', 'Failed to create sub-community. Error: ' . $errorBody);
            }
        } catch (ClientException $e) {
            $status = $e->getResponse()->getStatusCode();
            $body   = (string) $e->getResponse()->getBody();

            Log::error("DSpace DELETE ERROR", [
                'status' => $status,
                'body'   => $body
            ]);

            // unauthorized
            if ($status == 401) {
                session()->flush();
                return redirect()->route('auth-login-basic')
                    ->with('error', 'Authentication expired. Please login again.');
            }

            return redirect()->back()->with('error', 'Failed to delete admin');
        } catch (\Exception $e) {
            return response()->json(['error' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }

    // Edit Sub Community Function
    public function editSubCommunity(Request $request)
    {
        // Get ID and new name from the request
        $id = $request->input('id');

        $name = $request->input('subCommunityName');
        $title = $this->getTitleFieldId();
        // Update the database
        DB::table('metadatavalue')
            ->where('dspace_object_id', $id)
            ->where('metadata_field_id', $title) // Ensure correct field
            ->update(['text_value' => $name]);
        // Redirect back with success message
        return redirect()->back()->with('success', 'Sub Community updated successfully!');
    }

    // Delete Sub Community Function
    public function deleteSubCommunity($id, DSpaceApiService $apiService)
    {
        try {
            $authorizationHeader = Session::get('api_token');

            $response = $apiService->delete('core/communities/' . $id, $authorizationHeader);


            if ($response->getStatusCode() === 204) {
                // Redirect to the desired view with a success message
                return redirect()->back()->with('success', 'Sub-Community deleted successfully');
            } else {
                // Redirect back with an error message or handle the error accordingly
                return redirect()->back()->with('error', 'Failed to delete process');
            }
        } catch (ClientException $e) {
            $status = $e->getResponse()->getStatusCode();
            $body   = (string) $e->getResponse()->getBody();

            Log::error("DSpace DELETE ERROR", [
                'status' => $status,
                'body'   => $body
            ]);

            // unauthorized
            if ($status == 401) {
                session()->flush();
                return redirect()->route('auth-login-basic')
                    ->with('error', 'Authentication expired. Please login again.');
            }

            return redirect()->back()->with('error', 'Failed to delete admin');
        } catch (\Exception $e) {
            return response()->json(['error' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }
}
