<?php

namespace App\Http\Controllers\pages;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Services\DSpaceApiService;


class CommunitiesController extends Controller
{

    protected $title;

    private function getTitleFieldId()
    {
        return DB::table('metadatafieldregistry')
            ->where('element', 'title')
            ->whereNull('qualifier')
            ->where('metadata_schema_id', 1)
            ->value('metadata_field_id');
    }

    // Index 
    public function index()
    {
        $title = $this->getTitleFieldId();
        // Fetch communities that are not child communities
        $data = DB::table('public.metadatavalue as md')
            ->join('community as c', 'md.dspace_object_id', '=', 'c.uuid')
            ->select('md.text_value', 'md.dspace_object_id')
            ->where('md.metadata_field_id', $title)
            ->whereNotIn('md.dspace_object_id', function ($query) {
                $query->select('child_comm_id')->from('public.community2community');
            })
            ->get();

        return view('content.pages.community', [
            'data' => $data,
            'count' => $data->count(),
        ]);
    }

    // Add new community
    public function addCommunity(Request $request, DSpaceApiService $apiService)
    {
        $request->validate([
            'name' => 'required|string|max:255',
        ]);

        $authToken = session('api_token');
        if (!$authToken) {
            return back()->with('error', 'Authorization token not found.');
        }

        $data = [
            'name' => $request->name,
            'metadata' => [
                'dc.title' => [[
                    'value' => $request->name,
                    'language' => null,
                    'authority' => null,
                    'confidence' => -1
                ]]
            ]
        ];

        try {
            $response = $apiService->post('core/communities', $data, $authToken);

            if ($response->getStatusCode() === 201) {
                return back()->with('success', 'Community created successfully!');
            }

            return back()->with('error', 'Failed to create community: ' . $response->getBody());
        } catch (\Exception $e) {
            return back()->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }

    // Edit Community name
    public function editCommunity(Request $request)
    {
        $request->validate([
            'id' => 'required|uuid',
            'community_name' => 'required|string|max:255',
        ]);
        $title = $this->getTitleFieldId();
        $updated = DB::table('metadatavalue as md')
            ->join('community as c', 'md.dspace_object_id', '=', 'c.uuid')
            ->where('md.metadata_field_id', $title)
            ->where('md.dspace_object_id', $request->id)
            ->update(['text_value' => $request->community_name]);

        return $updated
            ? back()->with('success', 'Community updated successfully!')
            : back()->with('error', 'Failed to update community.');
    }

    // Delete Community 
    public function delete($id, DSpaceApiService $apiService)
    {

        $authToken = session('api_token');
        if (!$authToken) {
            return redirect()->route('auth-login-basic')->with('error', 'Authorization token not found.');
        }

        try {
            $response = $apiService->delete('core/communities/' . $id, $authToken);

            if ($response->getStatusCode() === 204) {
                return redirect()->route('community.index')->with('success', 'Community deleted successfully');
            }

            return back()->with('error', 'Failed to delete community: ' . $response->getBody());
        } catch (\Exception $e) {
            return back()->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }
}
