<?php

namespace App\Http\Controllers\pages;

use App\Http\Controllers\Controller;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use phpseclib3\Net\SSH2;
use GuzzleHttp\Client;

ini_set('max_execution_time', 3600); // 3600 seconds = 60 minutes

class DataIntegrationController extends Controller
{

    private $host;
    private $port;
    private $username;
    private $password;
    protected $scopus_author_id;
    protected $contributor;
    protected $research_id;
    protected $entitytype;

    // Constructor
    public function __construct()
    {
        $this->host = config('services.putty.host'); // Load from .env via config
        $this->port = config('services.putty.port');
        $this->username = config('services.putty.username');
        $this->password = config('services.putty.password');
        $this->scopus_author_id = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'scopus-author-id')
            ->value('metadata_field_id');
        $this->contributor = DB::table('metadatafieldregistry')
            ->where('element', 'contributor')
            ->where('qualifier', 'author')
            ->value('metadata_field_id');
        $this->entitytype = DB::table('metadatafieldregistry')
            ->where('element', 'entity')
            ->where('qualifier', 'type')
            ->value('metadata_field_id');
        $this->research_id = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'rid')
            ->value('metadata_field_id');
    }

    //index function
    public function index()
    {
        // Pass update_publication data to the view
        $data = DB::table('update_publications')->get();

        // Pass the data to the view
        return view('content.pages.data-integration', ['publications' => $data]);
    }

    // SheraUrl funcation this was added in cris 8
    public function sherpaUrl(Request $request)
    {
        $issnId = DB::table('public.metadatafieldregistry')
            ->where('qualifier', 'like', '%issn%')
            ->where('element', 'identifier')
            ->value('metadata_field_id');  // Return issn id

        $sherpaUrlId = DB::table('metadatafieldregistry')
            ->where('qualifier', 'LIKE', '%sherpaUrl%')
            ->value('metadata_field_id');  // Returns a single value instead of a collection

        $metadataRecords = DB::table('metadatavalue')
            ->where('metadata_field_id', $issnId)
            ->get();

        $text_value_pub = 'Update_Sherpa_URL';
        $update_count = 0;

        $client = new Client();  // Create a new Guzzle HTTP client instance

        foreach ($metadataRecords as $record) {
            $issn = $record->text_value;
            $uuid = $record->dspace_object_id;
            $apiUrl = "https://v2.sherpa.ac.uk/cgi/retrieve_by_id?item-type=publication&api-key=73FF124E-6275-11EE-B118-4DCE7D1EFE05&format=Json&identifier=" . $issn;

            try {
                $response = $client->get($apiUrl);
                $data = json_decode($response->getBody(), true);

                if (!empty($data['items']) && isset($data['items'][0]['system_metadata'])) {
                    $item = $data['items'][0];

                    if (isset($item['system_metadata']['publicly_visible']) && $item['system_metadata']['publicly_visible'] === "yes") {
                        if (isset($item['system_metadata']['uri'])) {
                            $sherpaUrl = $item['system_metadata']['uri']; // Take URI value

                            // Check if record exists
                            $existingRecord = DB::table('metadatavalue')
                                ->where('metadata_field_id', $sherpaUrlId)
                                ->where('dspace_object_id', $uuid)
                                ->first();

                            $update_count++;

                            if ($existingRecord) {
                                // Update existing record
                                DB::table('metadatavalue')
                                    ->where('metadata_field_id', $sherpaUrlId)
                                    ->where('dspace_object_id', $uuid)
                                    ->update(['text_value' => $sherpaUrl]);
                            } else {
                                // Insert new record
                                DB::table('metadatavalue')->insert([
                                    'metadata_field_id' => $sherpaUrlId,
                                    'text_value' => $sherpaUrl,
                                    'text_lang' => null,
                                    'place' => 0,
                                    'authority' => null,
                                    'confidence' => -1,
                                    'dspace_object_id' => $uuid,
                                    'security_level' => 0
                                ]);
                            }
                        }
                    }
                }
            } catch (\Exception $e) {
                // Log or handle errors as needed
                continue;
            }
        } // end of foreach

        // Check if text_value exists
        $exists = DB::table('update_publications')
            ->where('text_value', $text_value_pub)
            ->exists();

        if ($exists) {
            // If exists, update last_date only
            DB::table('update_publications')
                ->where('text_value', $text_value_pub)
                ->update([
                    'updated_count' => $update_count,
                    'last_date' => Carbon::today()
                ]);
        } else {
            // If not exists, insert new record
            DB::table('update_publications')->insert([
                'text_value' => $text_value_pub,
                'updated_count' => $update_count,
                'last_date' => Carbon::today()
            ]);
        }

        session()->flash('success', 'Update Sherpa URL successfully!');
        return response()->json(['success' => true]);
    }

    //  crossRefCiatation funcation this was added in cris 8
    public function crossRefCiatation(Request $request)
    {
        // Get metadata_field_id for 'doi'
        $metadataFieldId_doi = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'LIKE', '%doi%')
            ->value('metadata_field_id');

        // Get metadata_field_id for 'crossref_citation'
        $metadataFieldId_crossref = DB::table('metadatafieldregistry')
            ->where('qualifier', 'LIKE', '%crossref_citation%')
            ->value('metadata_field_id');

        // Get DOI records
        $doi_records = DB::table('metadatavalue')
            ->select('text_value', 'dspace_object_id')
            ->where('metadata_field_id', $metadataFieldId_doi)
            ->get();

        $text_value_pub = 'Update_Crossref_Citation';
        $update_count = 0;

        $client = new \GuzzleHttp\Client();  // Guzzle client instance

        foreach ($doi_records as $doi_record) {
            $doi = $doi_record->text_value;
            $url = "http://api.crossref.org/v1/works/http://dx.doi.org/{$doi}";

            try {
                $response = $client->get($url);
                $statusCode = $response->getStatusCode();

                if ($statusCode === 200) {
                    $json_data = json_decode($response->getBody(), true);
                    $data_parts = (array) $json_data['message'];
                    $crossref_citation = $data_parts['is-referenced-by-count'] ?? 0;

                    // Check if record exists
                    $citationExists = DB::table('metadatavalue')
                        ->where('metadata_field_id', $metadataFieldId_crossref)
                        ->where('dspace_object_id', $doi_record->dspace_object_id)
                        ->exists();

                    $update_count++;

                    if ($citationExists) {
                        // Update existing record
                        DB::table('metadatavalue')
                            ->where('dspace_object_id', $doi_record->dspace_object_id)
                            ->where('metadata_field_id', $metadataFieldId_crossref)
                            ->update([
                                'text_value' => $crossref_citation,
                                'text_lang' => null,
                                'place' => 0,
                                'authority' => null,
                                'confidence' => -1,
                                'security_level' => null,
                            ]);
                    } else {
                        // Insert new record
                        DB::table('metadatavalue')->insert([
                            'metadata_field_id' => $metadataFieldId_crossref,
                            'text_value' => $crossref_citation,
                            'text_lang' => null,
                            'place' => 0,
                            'authority' => null,
                            'confidence' => -1,
                            'dspace_object_id' => $doi_record->dspace_object_id,
                            'security_level' => null,
                        ]);
                    }
                }
            } catch (\GuzzleHttp\Exception\ClientException $e) {
                // Check if it's a 404 error
                if ($e->getResponse() && $e->getResponse()->getStatusCode() === 404) {
                    // Skip this DOI and continue to the next one
                    continue;
                } else {
                    // Other client errors
                    session()->flash('error', 'Client error occurred: ' . $e->getMessage());
                    return response()->json(['success' => false, 'error' => $e->getMessage()]);
                }
            } catch (\Exception $e) {
                // Handle other exceptions
                session()->flash('error', 'An error occurred: ' . $e->getMessage());
                return response()->json(['success' => false, 'error' => $e->getMessage()]);
            }
        }

        // Update publications table
        $exists = DB::table('update_publications')
            ->where('text_value', $text_value_pub)
            ->exists();

        if ($exists) {
            DB::table('update_publications')
                ->where('text_value', $text_value_pub)
                ->update([
                    'updated_count' => $update_count,
                    'last_date' => \Carbon\Carbon::today()
                ]);
        } else {
            DB::table('update_publications')->insert([
                'text_value' => $text_value_pub,
                'updated_count' => $update_count,
                'last_date' => \Carbon\Carbon::today(),
            ]);
        }

        session()->flash('success', 'Update CrossRef Citation successfully!');
        return response()->json(['success' => true]);
    }

    // unpaywall funcation this was added in cris 8
    public function unpaywall(Request $request)
    {
        try {
            // Fetch metadata field IDs
            $metadataFieldId_url = DB::table('metadatafieldregistry')
                ->where('element', 'identifier')
                ->where('qualifier', 'LIKE', '%upurl%')
                ->value('metadata_field_id');

            $metadataFieldId_color = DB::table('metadatafieldregistry')
                ->where('qualifier', 'LIKE', '%unpaywall%')
                ->value('metadata_field_id');

            // Fetch DOI records
            $doiRecords = DB::table('metadatavalue')
                ->where('metadata_field_id', function ($query) {
                    $query->select('metadata_field_id')
                        ->from('public.metadatafieldregistry')
                        ->where('element', 'identifier')
                        ->where('qualifier', 'LIKE', '%doi%');
                })
                ->get();

            $text_value_pub = 'Update_Unpaywall';
            $update_count = 0;
            $client = new Client();  // Guzzle client

            foreach ($doiRecords as $record) {
                if (!empty($record->text_value)) {
                    $doi = $record->text_value;
                    $uuid = $record->dspace_object_id;
                    $url = "https://api.oadoi.org/{$doi}?email=xyz@abc.com";

                    try {
                        $response = $client->get($url);
                        if ($response->getStatusCode() === 200) {
                            $json_data = json_decode($response->getBody(), true);
                            $oa_status = $json_data['oa_status'] ?? 'closed';
                            $oa_status = ($oa_status === 'closed') ? 'close' : $oa_status;
                            $url_for_landing_page = ($oa_status !== 'closed')
                                ? ($json_data['best_oa_location']['url'] ?? null)
                                : null;

                            // Update or insert URL record
                            $existingRecord = DB::table('metadatavalue')
                                ->where('dspace_object_id', $uuid)
                                ->where('metadata_field_id', $metadataFieldId_url)
                                ->first();

                            $update_count++;

                            if ($existingRecord) {
                                DB::table('metadatavalue')
                                    ->where('dspace_object_id', $uuid)
                                    ->where('metadata_field_id', $metadataFieldId_url)
                                    ->update(['text_value' => $url_for_landing_page]);
                            } else {
                                DB::table('metadatavalue')->insert([
                                    'metadata_field_id' => $metadataFieldId_url,
                                    'text_value' => $url_for_landing_page,
                                    'text_lang' => null,
                                    'place' => 0,
                                    'authority' => null,
                                    'confidence' => -1,
                                    'dspace_object_id' => $uuid,
                                    'security_level' => null,
                                ]);
                            }

                            // Update or insert OA status record
                            $existingRecord_color = DB::table('metadatavalue')
                                ->where('dspace_object_id', $uuid)
                                ->where('metadata_field_id', $metadataFieldId_color)
                                ->first();

                            if ($existingRecord_color) {
                                DB::table('metadatavalue')
                                    ->where('dspace_object_id', $uuid)
                                    ->where('metadata_field_id', $metadataFieldId_color)
                                    ->update(['text_value' => $oa_status]);
                            } else {
                                DB::table('metadatavalue')->insert([
                                    'metadata_field_id' => $metadataFieldId_color,
                                    'text_value' => $oa_status,
                                    'text_lang' => null,
                                    'place' => 0,
                                    'authority' => null,
                                    'confidence' => -1,
                                    'dspace_object_id' => $uuid,
                                    'security_level' => null,
                                ]);
                            }
                        } else {
                            // Log unexpected status code or handle accordingly
                        }
                    } catch (\Exception $e) {
                        // Log::error("API request failed for DOI: {$doi} - " . $e->getMessage());
                    }
                }
            }

            // Update publications table
            $exists = DB::table('update_publications')
                ->where('text_value', $text_value_pub)
                ->exists();

            if ($exists) {
                DB::table('update_publications')
                    ->where('text_value', $text_value_pub)
                    ->update(['updated_count' => $update_count, 'last_date' => Carbon::today()]);
            } else {
                DB::table('update_publications')->insert([
                    'text_value' => $text_value_pub,
                    'updated_count' => $update_count,
                    'last_date' => Carbon::today(),
                ]);
            }

            session()->flash('success', 'Updated Unpaywall URLs successfully!');
            return response()->json(['success' => true], 200);
        } catch (\Exception $e) {
            // Log::error("Script execution failed: " . $e->getMessage());
            return response()->json(['error' => 'An error occurred while updating records.'], 500);
        }
    }

    // importScopusCitation via API
    public function importScopusCitation(Request $request)
    {
        // Fetch metadata_field_id dynamically
        $affiliationId = DB::table('metadatafieldregistry')
            ->where('element', 'affiliation')
            ->where('qualifier', 'id')
            ->value('metadata_field_id');

        $scopusAuthorId = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'scopus-author-id')
            ->value('metadata_field_id');

        $doiId = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'doi')
            ->value('metadata_field_id');

        $metadataFieldId_scopus_citation = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'scopus_citation')
            ->value('metadata_field_id');

        $textValue = DB::table('metadatavalue')
            ->where('metadata_field_id', $affiliationId)
            ->select('text_value', 'dspace_object_id')
            ->first();

        $affId = $textValue->text_value ?? null;

        $text_value_pub = 'Import_Scopus_Citation';
        $update_count = 0;

        $apiKey = "0bb7c13cdcb6419d3882c2cc63859a97";
        $perPage = 25;

        $client = new Client();

        $scopusIds = DB::table('metadatavalue as mv_outer')
            ->select('mv_outer.text_value', 'mv_outer.dspace_object_id')
            ->whereIn('mv_outer.dspace_object_id', function ($query) {
                $query->select('c2i.item_id')
                    ->from('collection2item as c2i')
                    ->join('metadatavalue as mv', 'c2i.collection_id', '=', 'mv.dspace_object_id')
                    ->join('public.collection as col', 'mv.dspace_object_id', '=', 'col.uuid')
                    ->where('mv.metadata_field_id', 7)
                    ->where('mv.text_value', 'Person');
            })
            ->where('mv_outer.metadata_field_id', $scopusAuthorId)
            ->get();

        foreach ($scopusIds as $scopusIdObj) {
            $scopusId = $scopusIdObj->text_value;

            $url = "https://api.elsevier.com/content/search/scopus?query=af-id({$affId})+and+au-id({$scopusId})&APIKey={$apiKey}&httpAccept=application/json&start=0&view=COMPLETE&count={$perPage}";

            $response = $client->request('GET', $url);

            if ($response->getStatusCode() === 200) {
                $json_data = json_decode($response->getBody(), true);

                $totalResults = $json_data['search-results']['opensearch:totalResults'] ?? 0;
                $itemsPerPage = $json_data['search-results']['opensearch:itemsPerPage'] ?? 0;
                $entries = [];

                foreach ($json_data['search-results']['entry'] as $entry) {
                    $prism_doi = $entry['prism:doi'] ?? null;
                    $citedbyCount = $entry['citedby-count'] ?? 0;
                    if ($prism_doi) {
                        $entries[$prism_doi] = (string) $citedbyCount;
                    }
                }

                $remainingRequests = ceil(($totalResults - $itemsPerPage) / $perPage);

                for ($j = 1; $j <= $remainingRequests; $j++) {
                    $start = $j * $perPage;
                    $url = "https://api.elsevier.com/content/search/scopus?query=af-id({$affId})+and+au-id({$scopusId})&APIKey={$apiKey}&httpAccept=application/json&start={$start}&view=COMPLETE&count={$perPage}";

                    $response = $client->request('GET', $url);

                    if ($response->getStatusCode() === 200) {
                        $additionalData = json_decode($response->getBody(), true);

                        foreach ($additionalData['search-results']['entry'] as $entry) {
                            $prism_doi = $entry['prism:doi'] ?? null;
                            $citedbyCount = $entry['citedby-count'] ?? 0;
                            if ($prism_doi) {
                                $entries[$prism_doi] = $citedbyCount;
                            }
                        }
                    }
                }

                foreach ($entries as $doi => $citedbyCount) {
                    $dspaceObjectId = DB::table('metadatavalue')
                        ->where('text_value', $doi)
                        ->where('metadata_field_id', $doiId)
                        ->value('dspace_object_id');

                    if ($dspaceObjectId) {
                        $update_count++;
                        $existingRecord = DB::table('metadatavalue')
                            ->where('dspace_object_id', $dspaceObjectId)
                            ->where('metadata_field_id', $metadataFieldId_scopus_citation)
                            ->first();

                        if ($existingRecord) {
                            DB::table('metadatavalue')
                                ->where('dspace_object_id', $dspaceObjectId)
                                ->where('metadata_field_id', $metadataFieldId_scopus_citation)
                                ->update([
                                    'text_value' => $citedbyCount,
                                    'text_lang' => null,
                                    'place' => 0,
                                    'authority' => null,
                                    'confidence' => -1,
                                    'security_level' => null,
                                ]);
                        } else {
                            DB::table('metadatavalue')->insert([
                                'metadata_field_id' => $metadataFieldId_scopus_citation,
                                'text_value' => $citedbyCount,
                                'text_lang' => null,
                                'place' => 0,
                                'authority' => null,
                                'confidence' => -1,
                                'dspace_object_id' => $dspaceObjectId,
                                'security_level' => null,
                            ]);
                        }
                    }
                }
            }
        }

        $exists = DB::table('update_publications')
            ->where('text_value', $text_value_pub)
            ->exists();

        if ($exists) {
            DB::table('update_publications')
                ->where('text_value', $text_value_pub)
                ->update(['updated_count' => $update_count, 'last_date' => Carbon::today()]);
        } else {
            DB::table('update_publications')->insert([
                'text_value' => $text_value_pub,
                'updated_count' => $update_count,
                'last_date' => Carbon::today(),
            ]);
        }

        session()->flash('success', 'Import Scopus Citation successfully!');
        return response()->json(['success' => true]);
    }

    // updateScopusCitation via API for Publication
    public function updateScopusCitation(Request $request)
    {
        try {
            // 🔹 SSH Configuration
            $host = $this->host;
            $port = $this->port;
            $username = $this->username;
            $password = $this->password;

            $text_value_pub = 'import_update_scopus_publications_citation';
            $update_count = 0;
            // 🔹 Command to run
            $command = 'cd /dspace/bin && echo ' . escapeshellarg($password) . ' | sudo -S ./dspace update-metrics -s scopus -publication CITATION';

            // Attempt SSH execution using phpseclib3
            Log::info('Attempting SSH connection to run Citation');
            $ssh = new SSH2($host, $port);

            if (!$ssh->login($username, $password)) {
                throw new \Exception("SSH login failed.");
            }

            $output = $ssh->exec($command);

            // Log and return response
            Log::info('SSH Command Output', ['output' => $output]);


            if (!empty($output)) {
                // Log::info('SSH Command Output', ['output' => $output]);

                $update_count = DB::table('cris_metrics')
                    ->where('metrictype', 'scopusCitation')
                    ->whereDate('acquisitiondate', Carbon::today()->format('Y-m-d'))
                    ->count();

                // Check if text_value exists
                $exists = DB::table('update_publications')
                    ->where('text_value', $text_value_pub)
                    ->exists();

                // dd($exists);


                if ($exists) {
                    // If exists, update last_date only
                    DB::table('update_publications')
                        ->where('text_value', $text_value_pub)
                        ->update([
                            'updated_count' => $update_count,
                            'last_date' => Carbon::today()
                        ]);
                } else {
                    // If not exists, insert new record
                    DB::table('update_publications')->insert([
                        'text_value' => $text_value_pub,
                        'updated_count' => $update_count,
                        'last_date' => Carbon::today()
                    ]);
                }

                session()->flash('success', 'Scopus citation Updated Successfully!');
            } else {
                session()->flash('error', 'No output received from SSH command.');
            }

            return response()->json(['success' => true]);
        } catch (Exception $e) {
            Log::error('SSH Connection Error: ' . $e->getMessage());
            session()->flash('error', 'SSH Connection Failed: ' . $e->getMessage());
        }
    }

    // update solar index through Command
    public function updateSolorIndex(Request $request)
    {
        try {
            // 🔹 SSH Configuration
            $host = $this->host;
            $port = $this->port;
            $username = $this->username;
            $password = $this->password;

            // 🔹 Command to run
            $command = 'cd /dspace/bin && echo ' . escapeshellarg($password) . ' | sudo -S ./dspace index-discovery -f';

            // Attempt SSH execution using phpseclib3
            Log::info('Attempting SSH connection to run index-discovery');
            $ssh = new SSH2($host, $port);

            if (!$ssh->login($username, $password)) {
                throw new \Exception("SSH login failed.");
            }

            $output = $ssh->exec($command);

            // Log and return response
            Log::info('SSH Command Output', ['output' => $output]);


            if (!empty($output)) {
                // Log::info('SSH Command Output', ['output' => $output]);
                session()->flash('success', 'Solr Index Updated Successfully!');
            } else {
                session()->flash('error', 'No output received from SSH command.');
            }

            return response()->json(['success' => true]);
        } catch (Exception $e) {
            Log::error('SSH Connection Error: ' . $e->getMessage());
            session()->flash('error', 'SSH Connection Failed: ' . $e->getMessage());
        }
    }

    // update goto h index through Command
    public function updateH_Index(Request $request)
    {
        try {
            // 🔹 SSH Configuration
            $host = $this->host;
            $port = $this->port;
            $username = $this->username;
            $password = $this->password;

            // 🔹 Command to run
            $command = 'cd /dspace/bin && echo ' . escapeshellarg($password) . ' | sudo -S ./dspace update-metrics -s scopus-person -p H_INDEX';

            // Attempt SSH execution using phpseclib3
            Log::info('Attempting SSH connection to run H-index');
            $ssh = new SSH2($host, $port);

            if (!$ssh->login($username, $password)) {
                throw new \Exception("SSH login failed.");
            }

            $output = $ssh->exec($command);

            // Log and return response
            Log::info('SSH Command Output', ['output' => $output]);


            if (!empty($output)) {
                // Log::info('SSH Command Output', ['output' => $output]);
                session()->flash('success', 'H-index Updated Successfully!');
            } else {
                session()->flash('error', 'No output received from SSH command.');
            }

            return response()->json(['success' => true]);
        } catch (Exception $e) {
            Log::error('SSH Connection Error: ' . $e->getMessage());
            session()->flash('error', 'SSH Connection Failed: ' . $e->getMessage());
        }
    }

    // update citation through Command for Person
    public function updateCitation(Request $request)
    {
        try {
            // 🔹 SSH Configuration
            $host = $this->host;
            $port = $this->port;
            $username = $this->username;
            $password = $this->password;


            // 🔹 Command to run
            $command = 'cd /dspace/bin && echo ' . escapeshellarg($password) . ' | sudo -S ./dspace update-metrics -s "scopus-person" -p "CITATION" ';

            // Attempt SSH execution using phpseclib3
            Log::info('Attempting SSH connection to run Citation');
            $ssh = new SSH2($host, $port);

            if (!$ssh->login($username, $password)) {
                throw new \Exception("SSH login failed.");
            }

            $output = $ssh->exec($command);

            // Log and return response
            Log::info('SSH Command Output', ['output' => $output]);


            if (!empty($output)) {
                // Log::info('SSH Command Output', ['output' => $output]);
                session()->flash('success', 'Citation Updated Successfully!');
            } else {
                session()->flash('error', 'No output received from SSH command.');
            }

            return response()->json(['success' => true]);
        } catch (Exception $e) {
            Log::error('SSH Connection Error: ' . $e->getMessage());
            session()->flash('error', 'SSH Connection Failed: ' . $e->getMessage());
        }
    }

    // update co author through Command
    public function updateCoauthor(Request $request)
    {
        try {
            // 🔹 SSH Configuration
            $host = $this->host;
            $port = $this->port;
            $username = $this->username;
            $password = $this->password;

            // 🔹 Command to run
            $command = 'cd /dspace/bin && echo ' . escapeshellarg($password) . ' | sudo -S ./dspace update-metrics -s scopus-person -p COAUTHOR';

            // Attempt SSH execution using phpseclib3
            Log::info('Attempting SSH connection to run Co-Author');
            $ssh = new SSH2($host, $port);

            if (!$ssh->login($username, $password)) {
                throw new \Exception("SSH login failed.");
            }

            $output = $ssh->exec($command);

            // Log and return response
            Log::info('SSH Command Output', ['output' => $output]);


            if (!empty($output)) {
                // Log::info('SSH Command Output', ['output' => $output]);
                session()->flash('success', 'Co-authors Updated Successfully!');
            } else {
                session()->flash('error', 'No output received from SSH command.');
            }

            return response()->json(['success' => true]);
        } catch (Exception $e) {
            Log::error('SSH Connection Error: ' . $e->getMessage());
            session()->flash('error', 'SSH Connection Failed: ' . $e->getMessage());
        }
    }

    // update documentes through Command
    public function updateDocumentes(Request $request)
    {
        try {
            // 🔹 SSH Configuration
            $host = $this->host;
            $port = $this->port;
            $username = $this->username;
            $password = $this->password;

            // 🔹 Command to run
            $command = 'cd /dspace/bin && echo ' . escapeshellarg($password) . ' | sudo -S ./dspace update-metrics -s scopus-person -p DOCUMENT';

            // Attempt SSH execution using phpseclib3
            Log::info('Attempting SSH connection to run Document');
            $ssh = new SSH2($host, $port);

            if (!$ssh->login($username, $password)) {
                throw new \Exception("SSH login failed.");
            }

            $output = $ssh->exec($command);

            // Log and return response
            Log::info('SSH Command Output', ['output' => $output]);


            if (!empty($output)) {
                // Log::info('SSH Command Output', ['output' => $output]);
                session()->flash('success', 'Documentes Updated Successfully!');
            } else {
                session()->flash('error', 'No output received from SSH command.');
            }

            return response()->json(['success' => true]);
        } catch (Exception $e) {
            Log::error('SSH Connection Error: ' . $e->getMessage());
            session()->flash('error', 'SSH Connection Failed: ' . $e->getMessage());
        }
    }

    // update cited through Command
    public function updateCited(Request $request)
    {
        try {
            // 🔹 SSH Configuration
            $host = $this->host;
            $port = $this->port;
            $username = $this->username;
            $password = $this->password;
            // 🔹 Command to run
            $command = 'cd /dspace/bin && echo ' . escapeshellarg($password) . ' | sudo -S ./dspace update-metrics -s scopus-person -p CITED';

            // Attempt SSH execution using phpseclib3
            Log::info('Attempting SSH connection to run Cited');
            $ssh = new SSH2($host, $port);

            if (!$ssh->login($username, $password)) {
                throw new \Exception("SSH login failed.");
            }

            $output = $ssh->exec($command);

            // Log and return response
            Log::info('SSH Command Output', ['output' => $output]);


            if (!empty($output)) {
                // Log::info('SSH Command Output', ['output' => $output]);
                session()->flash('success', 'Cited Updated Successfully!');
            } else {
                session()->flash('error', 'No output received from SSH command.');
            }

            return response()->json(['success' => true]);
        } catch (Exception $e) {
            Log::error('SSH Connection Error: ' . $e->getMessage());
            session()->flash('error', 'SSH Connection Failed: ' . $e->getMessage());
        }
    }

    // function for map person with orgunit with query
    public function mapPersonwithDepartment(Request $request)
    {
        // Your existing database queries remain the same
        $title = DB::table('metadatafieldregistry')
            ->where('element', 'title')
            ->whereNull('qualifier')
            ->where('metadata_schema_id', 1)
            ->value('metadata_field_id');

        $orgunitrel = DB::table('metadatafieldregistry')
            ->where('element', 'affiliation')
            ->where('qualifier', 'name')
            ->value('metadata_field_id');

        $entitytype = DB::table('metadatafieldregistry')
            ->where('element', 'entity')
            ->where('qualifier', 'type')
            ->value('metadata_field_id');

        $query = "
                    UPDATE metadatavalue p
                    SET authority = o.dspace_object_id,
                        confidence = 600
                    FROM metadatavalue o
                    WHERE p.text_value = o.text_value
                    AND o.metadata_field_id = $title  -- OrgUnit name field
                    AND p.metadata_field_id = $orgunitrel -- Person name field
                    AND p.dspace_object_id IN (
                        SELECT dspace_object_id FROM metadatavalue 
                        WHERE metadata_field_id = $entitytype AND text_value = 'Person'
                    )
                    AND o.dspace_object_id IN (
                        SELECT dspace_object_id FROM metadatavalue 
                        WHERE metadata_field_id = $entitytype AND text_value = 'OrgUnit'
                    );
                ";

        // Execute the query
        DB::statement($query);
        $affected = DB::affectingStatement($query);

        // Your existing H-index update
        $this->updateH_Index($request);

        // Store flash message for potential redirects
        if ($affected > 0) {
            session()->flash('success', "Person(s) mapped with OrgUnit successfully.");
        } else {
            session()->flash('error', 'No records were updated. Maybe all persons are already mapped.');
        }

        // Return JSON response to work with your JavaScript
        return response()->json(['success' => true]);
    }

    // function for map Publication Person With Person with Scopus id through Query
    public function mapPublicationwithPersonScop(Request $request)
    {
        $scopus_author_id = $this->scopus_author_id;
        $contributor = $this->contributor;
        $entitytype = $this->entitytype;

        $query = "
        UPDATE metadatavalue mv10
        SET 
          authority = person.dspace_object_id,
          confidence = 500
        FROM metadatavalue mv223
        JOIN (
            SELECT dspace_object_id, text_value
            FROM metadatavalue
            WHERE metadata_field_id = $scopus_author_id
              AND dspace_object_id IN (
                  SELECT dspace_object_id 
                  FROM metadatavalue 
                  WHERE metadata_field_id = $entitytype AND text_value = 'Person'
              )
        ) person
          ON mv223.text_value = person.text_value
        WHERE mv10.dspace_object_id = mv223.dspace_object_id
          AND mv10.place = mv223.place
          AND mv10.metadata_field_id = $contributor
          AND mv223.metadata_field_id = $scopus_author_id
          AND (mv10.authority IS NULL OR mv10.authority = '')
        ";

        // Execute the query (match the pattern in mapPersonwithDepartment)
        DB::statement($query);
        $affected = DB::affectingStatement($query);

        // Call updateSolorIndex (similar to how updateH_Index is called)
        $this->updateSolorIndex($request);

        // Store flash message for potential redirects (same pattern as mapPersonwithDepartment)
        if ($affected >= 0) {
            session()->flash('success', "Publication authors successfully mapped with Person based on Scopus Id.");
        } else {
            session()->flash('error', "No publication records were updated. Either already mapped or no match found.");
        }

        // Return JSON response to work with your JavaScript (match pattern)
        return response()->json(['success' => true]);
    }
}
