<?php

namespace App\Http\Controllers\pages;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class DepartmentController extends Controller
{
    protected $titleFieldId;
    protected $entityTypeId;
    protected $citedById;
    protected $unpaywallId;
    protected $scopusId;
    protected $wosId;
    protected $crossrefCitationId;
    protected $subjectScopusId;
    protected $subjectWosId;
    protected $typeId;
    protected $issueDateId;


    public function __construct()
    {

        $this->titleFieldId = DB::table('metadatafieldregistry')
            ->where('element', 'title')
            ->whereNull('qualifier')
            ->where('metadata_schema_id', 1)
            ->value('metadata_field_id');

        $this->entityTypeId = DB::table('metadatafieldregistry')
            ->where('element', 'entity')
            ->where('qualifier', 'type')
            ->value('metadata_field_id');

        $this->citedById = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'citedby')
            ->value('metadata_field_id');

        $this->unpaywallId = DB::table('metadatafieldregistry')
            ->where('element', 'venue')
            ->where('qualifier', 'unpaywall')
            ->value('metadata_field_id');

        $this->scopusId = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'scopus')
            ->value('metadata_field_id');

        $this->wosId = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'wos')
            ->value('metadata_field_id');

        $this->crossrefCitationId = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'crossref_citation')
            ->value('metadata_field_id');

        $this->subjectScopusId = DB::table('metadatafieldregistry')
            ->where('element', 'subject_scopus')
            // ->where('qualifier', 'scopus')
            ->value('metadata_field_id');

        $this->subjectWosId = DB::table('metadatafieldregistry')
            ->where('element', 'subject_wos')
            // ->where('qualifier', 'scopus')
            ->value('metadata_field_id');

        $this->typeId = DB::table('metadatafieldregistry')
            ->where('element', 'type')
            ->where('metadata_schema_id', function ($query) {
                $query->select('metadata_schema_id')
                    ->from('metadataschemaregistry')
                    ->where('short_id', 'dc')
                    ->limit(1);
            })
            ->value('metadata_field_id');

        $this->issueDateId = DB::table('metadatafieldregistry')
            ->where('element', 'date')
            ->where('qualifier', 'issued')
            ->value('metadata_field_id');
    }

    /**
     * Show the department counts page
     */
    public function index(Request $request)
    {
        return view('content.pages.departments-count');
    }

    /**
     * Return department data for DataTable (AJAX)
     */
    public function getDepartments(Request $request)
    {
        $yearFrom = $request->year_from;
        $yearTo   = $request->year_to;

        // -----------------------------------------
        // CASE 1: NO FILTER → Load from JSON
        // -----------------------------------------
        if (empty($yearFrom) || empty($yearTo)) {
            $jsonPath = storage_path('app/json/departments.json');

            if (!file_exists($jsonPath)) {
                return response()->json(['data' => []]);
            }

            $json = json_decode(file_get_contents($jsonPath), true);
            return response()->json(['data' => $json]);
        }

        // -----------------------------------------
        // CASE 2: FILTER APPLIED → Use CACHE + DB
        // -----------------------------------------

        $cacheKey = "departments_{$yearFrom}_{$yearTo}";

        $data = Cache::remember($cacheKey, now()->addHours(3), function () use ($yearFrom, $yearTo) {
            return $this->runFilteredQuer($yearFrom, $yearTo);
        });

        return response()->json(['data' => $data]);
    }


    public function updateData()
    {
        try {
            // 1️⃣ Run heavy query
            $data = $this->getHeavyQuery();

            // 2️⃣ Prepare JSON
            $json = json_encode($data, JSON_PRETTY_PRINT);

            if ($json === false) {
                throw new \Exception("JSON encoding failed");
            }

            // 3️⃣ Save safely (write to temp, then rename)
            $tempPath = storage_path('app/json/departments_temp.json');
            $finalPath = storage_path('app/json/departments.json');

            file_put_contents($tempPath, $json);

            // Atomic replace (prevents corrupted file)
            rename($tempPath, $finalPath);

            // 4️⃣ Flash success message
            session()->flash('success', 'Departments Count updated successfully!');

            return response()->json([
                'success' => true,
                'message' => session('success')
            ]);
        } catch (\Throwable $e) {

            // 5️⃣ Log the error for debugging
            Log::error('Departments update failed: ' . $e->getMessage());

            // 6️⃣ Return JSON error (so JavaScript can show message)
            return response()->json([
                'success' => false,
                'error'   => 'Failed to update Departments Count. Please try again.',
                'details' => $e->getMessage()  // Remove in production if needed
            ], 500);
        }
    }


    private function getHeavyQuery()
    {

        $titleFieldId = $this->titleFieldId;
        $entityTypeId = $this->entityTypeId;
        $citedById = $this->citedById;
        $unpaywallId = $this->unpaywallId;
        $scopusId = $this->scopusId;
        $wosId = $this->wosId;
        $crossrefCitationId = $this->crossrefCitationId;
        $subjectScopusId = $this->subjectScopusId;
        $subjectWosId = $this->subjectWosId;
        $typeId = $this->typeId;

        return DB::table('collection2item as c2i')

            /* ===== JOINS ===== */

            ->leftJoin('metadatavalue as exclude_meta', function ($join) use ($typeId) {
                $join->on('exclude_meta.dspace_object_id', '=', 'c2i.item_id')
                    ->where('exclude_meta.metadata_field_id', $typeId)
                    ->whereIn('exclude_meta.text_value', ['Ph.D.', 'M.Tech', 'M.Sc.', 'Poster Presented']);
            })

            ->leftJoin('metadatavalue as eprint_meta', function ($join) use ($typeId) {
                $join->on('eprint_meta.dspace_object_id', '=', 'c2i.item_id')
                    ->where('eprint_meta.metadata_field_id', $typeId)
                    ->where('eprint_meta.text_value', 'e-Print');
            })

            ->join('community2collection as c2c', 'c2i.collection_id', '=', 'c2c.collection_id')
            ->join('community as dept_comm', 'c2c.community_id', '=', 'dept_comm.uuid')
            ->join('metadatavalue as dept_meta', function ($join) use ($titleFieldId) {
                $join->on('dept_meta.dspace_object_id', '=', 'dept_comm.uuid')
                    ->where('dept_meta.metadata_field_id', $titleFieldId);
            })

            ->join('metadatavalue as pub_meta', function ($join) use ($entityTypeId) {
                $join->on('pub_meta.dspace_object_id', '=', 'c2i.item_id')
                    ->where('pub_meta.metadata_field_id', $entityTypeId)
                    ->where('pub_meta.text_value', 'Publication');
            })

            /* Citations subquery */
            ->leftJoin(
                DB::raw("(SELECT dspace_object_id, MAX(text_value::INTEGER) AS citation_count 
                         FROM metadatavalue 
                         WHERE metadata_field_id = $citedById 
                         GROUP BY dspace_object_id) AS pub_cite"),
                'pub_cite.dspace_object_id',
                '=',
                'c2i.item_id'
            )

            ->leftJoin('metadatavalue as oa_meta', function ($join)  use ($unpaywallId) {
                $join->on('oa_meta.dspace_object_id', '=', 'c2i.item_id')
                    ->where('oa_meta.metadata_field_id', $unpaywallId);
            })

            ->leftJoin('metadatavalue as scopus_meta', function ($join) use ($scopusId) {
                $join->on('scopus_meta.dspace_object_id', '=', 'c2i.item_id')
                    ->where('scopus_meta.metadata_field_id', $scopusId);
            })

            ->leftJoin('metadatavalue as wos_meta', function ($join) use ($wosId) {
                $join->on('wos_meta.dspace_object_id', '=', 'c2i.item_id')
                    ->where('wos_meta.metadata_field_id', $wosId);
            })

            ->leftJoin('metadatavalue as wos_cite_meta', function ($join) use ($crossrefCitationId) {
                $join->on('wos_cite_meta.dspace_object_id', '=', 'c2i.item_id')
                    ->where('wos_cite_meta.metadata_field_id', $crossrefCitationId);
            })

            ->leftJoin('metadatavalue as scopus_eng_meta', function ($join) use ($subjectScopusId) {
                $join->on('scopus_eng_meta.dspace_object_id', '=', 'c2i.item_id')
                    ->where('scopus_eng_meta.metadata_field_id', $subjectScopusId);
            })

            ->leftJoin('metadatavalue as wos_eng_meta', function ($join) use ($subjectWosId) {
                $join->on('wos_eng_meta.dspace_object_id', '=', 'c2i.item_id')
                    ->where('wos_eng_meta.metadata_field_id', $subjectWosId);
            })


            /* ===== WHERE CONDITION ===== */
            ->where(function ($query) {
                $query->whereNull('exclude_meta.dspace_object_id')
                    ->orWhere('eprint_meta.text_value', 'e-Print');
            })

            /* ===== SELECT FIELDS ===== */
            ->select([
                'dept_meta.text_value as department_name',

                DB::raw("COUNT(DISTINCT CASE WHEN eprint_meta.text_value IS NULL THEN c2i.item_id END) AS total_publications"),

                DB::raw("SUM(CASE WHEN eprint_meta.text_value IS NULL THEN COALESCE(pub_cite.citation_count, 0) END) AS total_publication_citations"),

                DB::raw("COUNT(DISTINCT CASE WHEN eprint_meta.text_value IS NULL AND oa_meta.text_value != 'close' THEN c2i.item_id END) AS open_access_publications"),

                DB::raw("SUM(CASE WHEN eprint_meta.text_value IS NULL AND oa_meta.text_value != 'close' THEN COALESCE(pub_cite.citation_count, 0) END) AS open_access_citations"),

                DB::raw("COUNT(DISTINCT CASE WHEN eprint_meta.text_value IS NULL AND scopus_meta.dspace_object_id IS NOT NULL THEN c2i.item_id END) AS scopus_publications"),

                DB::raw("SUM(CASE WHEN eprint_meta.text_value IS NULL AND scopus_meta.dspace_object_id IS NOT NULL THEN COALESCE(pub_cite.citation_count, 0) END) AS scopus_citations"),

                DB::raw("COUNT(DISTINCT CASE WHEN eprint_meta.text_value IS NULL AND wos_meta.dspace_object_id IS NOT NULL THEN c2i.item_id END) AS wos_publications"),

                DB::raw("SUM(CASE WHEN eprint_meta.text_value IS NULL AND wos_meta.dspace_object_id IS NOT NULL THEN COALESCE(wos_cite_meta.text_value::INTEGER, 0) END) AS wos_citations"),

                DB::raw("COUNT(DISTINCT CASE WHEN eprint_meta.text_value IS NULL AND scopus_meta.dspace_object_id IS NULL AND wos_meta.dspace_object_id IS NULL THEN c2i.item_id END) AS not_indexed_publications"),

                DB::raw("SUM(CASE WHEN eprint_meta.text_value IS NULL AND scopus_meta.dspace_object_id IS NULL AND wos_meta.dspace_object_id IS NULL THEN COALESCE(pub_cite.citation_count, 0) END) AS not_indexed_citations"),

                DB::raw("COUNT(DISTINCT CASE WHEN eprint_meta.text_value IS NULL AND scopus_eng_meta.text_value = 'ENG' THEN c2i.item_id END) AS scopus_eng_publications"),

                DB::raw("SUM(CASE WHEN eprint_meta.text_value IS NULL AND scopus_eng_meta.text_value = 'ENG' THEN COALESCE(pub_cite.citation_count, 0) END) AS scopus_eng_citations"),

                DB::raw("COUNT(DISTINCT CASE WHEN eprint_meta.text_value IS NULL AND wos_eng_meta.text_value = 'ENG' THEN c2i.item_id END) AS wos_eng_publications"),

                DB::raw("SUM(CASE WHEN eprint_meta.text_value IS NULL AND wos_eng_meta.text_value = 'ENG' THEN COALESCE(wos_cite_meta.text_value::INTEGER, 0) END) AS wos_eng_citations"),

                DB::raw("COUNT(DISTINCT CASE WHEN eprint_meta.text_value = 'e-Print' THEN c2i.item_id END) AS eprint_publications"),
            ])

            ->groupBy('dept_meta.text_value')

            ->orderBy('total_publications', 'DESC')

            ->get()
            ->toArray();
    }

    private function runFilteredQuer($yearFrom, $yearTo)
    {
        $yearFrom = $yearFrom;
        $yearTo   = $yearTo;
        $titleFieldId = $this->titleFieldId;
        $entityTypeId = $this->entityTypeId;
        $citedById = $this->citedById;
        $unpaywallId = $this->unpaywallId;
        $scopusId = $this->scopusId;
        $wosId = $this->wosId;
        $crossrefCitationId = $this->crossrefCitationId;
        $subjectScopusId = $this->subjectScopusId;
        $subjectWosId = $this->subjectWosId;
        $typeId = $this->typeId;
        $issueDateId = $this->issueDateId;

        return DB::table('collection2item as c2i')

            /* ===== JOINS ===== */
            // JOIN Issue Year
            ->leftJoin('metadatavalue as issue_date_meta', function ($join) use ($issueDateId){
                $join->on('issue_date_meta.dspace_object_id', '=', 'c2i.item_id')
                    ->where('issue_date_meta.metadata_field_id', $issueDateId);  // Issue date
            })
            ->leftJoin('metadatavalue as exclude_meta', function ($join) use ($typeId) {
                $join->on('exclude_meta.dspace_object_id', '=', 'c2i.item_id')
                    ->where('exclude_meta.metadata_field_id', $typeId)
                    ->whereIn('exclude_meta.text_value', ['Ph.D.', 'M.Tech', 'M.Sc.', 'Poster Presented']);
            })

            ->leftJoin('metadatavalue as eprint_meta', function ($join) use ($typeId) {
                $join->on('eprint_meta.dspace_object_id', '=', 'c2i.item_id')
                    ->where('eprint_meta.metadata_field_id', $typeId)
                    ->where('eprint_meta.text_value', 'e-Print');
            })

            ->join('community2collection as c2c', 'c2i.collection_id', '=', 'c2c.collection_id')
            ->join('community as dept_comm', 'c2c.community_id', '=', 'dept_comm.uuid')
            ->join('metadatavalue as dept_meta', function ($join) use ($titleFieldId) {
                $join->on('dept_meta.dspace_object_id', '=', 'dept_comm.uuid')
                    ->where('dept_meta.metadata_field_id', $titleFieldId);
            })

            ->join('metadatavalue as pub_meta', function ($join) use ($entityTypeId) {
                $join->on('pub_meta.dspace_object_id', '=', 'c2i.item_id')
                    ->where('pub_meta.metadata_field_id', $entityTypeId)
                    ->where('pub_meta.text_value', 'Publication');
            })

            /* Citations subquery */
            ->leftJoin(
                DB::raw("(SELECT dspace_object_id, MAX(text_value::INTEGER) AS citation_count 
                         FROM metadatavalue 
                         WHERE metadata_field_id = $citedById 
                         GROUP BY dspace_object_id) AS pub_cite"),
                'pub_cite.dspace_object_id',
                '=',
                'c2i.item_id'
            )

            ->leftJoin('metadatavalue as oa_meta', function ($join)  use ($unpaywallId) {
                $join->on('oa_meta.dspace_object_id', '=', 'c2i.item_id')
                    ->where('oa_meta.metadata_field_id', $unpaywallId);
            })

            ->leftJoin('metadatavalue as scopus_meta', function ($join) use ($scopusId) {
                $join->on('scopus_meta.dspace_object_id', '=', 'c2i.item_id')
                    ->where('scopus_meta.metadata_field_id', $scopusId);
            })

            ->leftJoin('metadatavalue as wos_meta', function ($join) use ($wosId) {
                $join->on('wos_meta.dspace_object_id', '=', 'c2i.item_id')
                    ->where('wos_meta.metadata_field_id', $wosId);
            })

            ->leftJoin('metadatavalue as wos_cite_meta', function ($join) use ($crossrefCitationId) {
                $join->on('wos_cite_meta.dspace_object_id', '=', 'c2i.item_id')
                    ->where('wos_cite_meta.metadata_field_id', $crossrefCitationId);
            })

            ->leftJoin('metadatavalue as scopus_eng_meta', function ($join) use ($subjectScopusId) {
                $join->on('scopus_eng_meta.dspace_object_id', '=', 'c2i.item_id')
                    ->where('scopus_eng_meta.metadata_field_id', $subjectScopusId);
            })

            ->leftJoin('metadatavalue as wos_eng_meta', function ($join) use ($subjectWosId) {
                $join->on('wos_eng_meta.dspace_object_id', '=', 'c2i.item_id')
                    ->where('wos_eng_meta.metadata_field_id', $subjectWosId);
            })


            /* ===== WHERE CONDITION ===== */
            ->where(function ($query) {
                $query->whereNull('exclude_meta.dspace_object_id')
                    ->orWhere('eprint_meta.text_value', 'e-Print');
            })
            // ✔ YEAR FILTER
            ->whereNotNull('issue_date_meta.text_value')
            ->whereRaw("issue_date_meta.text_value ~ '^[0-9]{4}-(0[1-9]|1[0-2])-(0[1-9]|[12][0-9]|3[01])$'")
            ->whereRaw("LEFT(issue_date_meta.text_value, 4)::INTEGER BETWEEN ? AND ?", [
                $yearFrom,
                $yearTo
            ])

            /* ===== SELECT FIELDS ===== */
            ->select([
                'dept_meta.text_value as department_name',

                DB::raw("COUNT(DISTINCT CASE WHEN eprint_meta.text_value IS NULL THEN c2i.item_id END) AS total_publications"),

                DB::raw("SUM(CASE WHEN eprint_meta.text_value IS NULL THEN COALESCE(pub_cite.citation_count, 0) END) AS total_publication_citations"),

                DB::raw("COUNT(DISTINCT CASE WHEN eprint_meta.text_value IS NULL AND oa_meta.text_value != 'close' THEN c2i.item_id END) AS open_access_publications"),

                DB::raw("SUM(CASE WHEN eprint_meta.text_value IS NULL AND oa_meta.text_value != 'close' THEN COALESCE(pub_cite.citation_count, 0) END) AS open_access_citations"),

                DB::raw("COUNT(DISTINCT CASE WHEN eprint_meta.text_value IS NULL AND scopus_meta.dspace_object_id IS NOT NULL THEN c2i.item_id END) AS scopus_publications"),

                DB::raw("SUM(CASE WHEN eprint_meta.text_value IS NULL AND scopus_meta.dspace_object_id IS NOT NULL THEN COALESCE(pub_cite.citation_count, 0) END) AS scopus_citations"),

                DB::raw("COUNT(DISTINCT CASE WHEN eprint_meta.text_value IS NULL AND wos_meta.dspace_object_id IS NOT NULL THEN c2i.item_id END) AS wos_publications"),

                DB::raw("SUM(CASE WHEN eprint_meta.text_value IS NULL AND wos_meta.dspace_object_id IS NOT NULL THEN COALESCE(wos_cite_meta.text_value::INTEGER, 0) END) AS wos_citations"),

                DB::raw("COUNT(DISTINCT CASE WHEN eprint_meta.text_value IS NULL AND scopus_meta.dspace_object_id IS NULL AND wos_meta.dspace_object_id IS NULL THEN c2i.item_id END) AS not_indexed_publications"),

                DB::raw("SUM(CASE WHEN eprint_meta.text_value IS NULL AND scopus_meta.dspace_object_id IS NULL AND wos_meta.dspace_object_id IS NULL THEN COALESCE(pub_cite.citation_count, 0) END) AS not_indexed_citations"),

                DB::raw("COUNT(DISTINCT CASE WHEN eprint_meta.text_value IS NULL AND scopus_eng_meta.text_value = 'ENG' THEN c2i.item_id END) AS scopus_eng_publications"),

                DB::raw("SUM(CASE WHEN eprint_meta.text_value IS NULL AND scopus_eng_meta.text_value = 'ENG' THEN COALESCE(pub_cite.citation_count, 0) END) AS scopus_eng_citations"),

                DB::raw("COUNT(DISTINCT CASE WHEN eprint_meta.text_value IS NULL AND wos_eng_meta.text_value = 'ENG' THEN c2i.item_id END) AS wos_eng_publications"),

                DB::raw("SUM(CASE WHEN eprint_meta.text_value IS NULL AND wos_eng_meta.text_value = 'ENG' THEN COALESCE(wos_cite_meta.text_value::INTEGER, 0) END) AS wos_eng_citations"),

                DB::raw("COUNT(DISTINCT CASE WHEN eprint_meta.text_value = 'e-Print' THEN c2i.item_id END) AS eprint_publications"),
            ])

            ->groupBy('dept_meta.text_value')

            ->orderBy('total_publications', 'DESC')

            ->get()
            ->toArray();
    }
}
