<?php

namespace App\Http\Controllers\pages;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Services\DSpaceApiService;
use GuzzleHttp\Exception\ClientException;
use Illuminate\Support\Facades\Log;

class EpersonController extends Controller
{

    // index
    public function index()
    {
        $filePath = storage_path('app/json/epersons.json');

        $data = [];

        // Check if the file exists
        if (file_exists($filePath)) {
            // Read the JSON file
            $fileContent = file_get_contents($filePath);

            // Decode the JSON content to an array
            $data = json_decode($fileContent, true);
        }

        // Pass the data to the view
        return view('content.pages.eperson', ['eperson' => $data]);
    }

    // Admin list update function
    public function eperson(Request $request)
    {
        if ($request->isMethod('get')) {
            if (session()->has('api_token')) {
                $this->createJsonFile();
                session()->flash('success', 'Admin list Updated Successfully!');

                return response()->json(['success' => session('success')]);
            } else {
                return response()->json(['error' => 'Failed to Update Eperson'], 400);
            }
        }
    }

    // Admin view 
    public function view(Request $request, $uuid)
    {
        try {
            // Fetch the specific eperson from the database
            $eperson = DB::table('eperson as e')
                ->join('metadatavalue as m', 'e.uuid', '=', 'm.dspace_object_id')
                ->join('epersongroup2eperson as ege', 'e.uuid', '=', 'ege.eperson_id')
                ->join('epersongroup as eg', 'eg.uuid', '=', 'ege.eperson_group_id')
                ->where('e.uuid', $uuid)
                ->select(
                    'e.uuid',
                    'e.email',
                    DB::raw("CONCAT(
                    MAX(CASE WHEN m.metadata_field_id = 1 THEN m.text_value ELSE NULL END), 
                    ' ', 
                    MAX(CASE WHEN m.metadata_field_id = 2 THEN m.text_value ELSE NULL END)
                ) AS concatenated_values"),
                    'eg.name'
                )
                ->groupBy('e.uuid', 'e.email', 'eg.name')
                ->first();

            if (!$eperson) {
                return response()->json(['error' => 'Admin not found.'], 404);
            }

            // Return the data as HTML (for modal)
            $html = view('content.pages.view-eperson', compact('eperson'))->render();

            return response()->json(['html' => $html]);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Something went wrong.'], 500);
        }
    }

    // createJsonFile
    public function createJsonFile()
    {
        // Read totalEPerson count from JSON file
        $filePath = storage_path('app/json/epersons.json');

        // Query the database to get the required data
        $epersons = DB::table('eperson as e')
            ->join('metadatavalue as m', 'e.uuid', '=', 'm.dspace_object_id')
            ->join('epersongroup2eperson as ege', 'e.uuid', '=', 'ege.eperson_id')
            ->join('epersongroup as eg', 'eg.uuid', '=', 'ege.eperson_group_id')
            ->whereIn('m.metadata_field_id', [1, 2])
            ->select(
                'e.uuid',
                'e.email',
                DB::raw("CONCAT(
           MAX(CASE WHEN m.metadata_field_id = 1 THEN m.text_value ELSE NULL END), 
           ' ', 
           MAX(CASE WHEN m.metadata_field_id = 2 THEN m.text_value ELSE NULL END)
       ) AS concatenated_values"),
                'eg.name'
            )
            ->groupBy('e.uuid', 'e.email', 'eg.name')
            ->get();

        $allData = $epersons->toArray();

        file_put_contents($filePath, json_encode($allData, JSON_PRETTY_PRINT));
    }

    // Delete Admin
    public function delete($id, DSpaceApiService $apiService)
    {
        try {
            $authorizationHeader = session()->get('api_token');

            if (!$authorizationHeader) {
                session()->flush();
                return redirect()->route('auth-login-basic')->with('error', 'Session expired. Please login again.');
            }

            // $response = $apiService->delete('eperson/epersons/' . $id, $authorizationHeader);
            $response = $apiService->delete('eperson/epersons/' . $id, $authorizationHeader);
            if ($response->getStatusCode() === 204) {
                $this->createJsonFile();
                return redirect()->route('eperson.index')->with('success', 'Admin deleted successfully');
            }

            return redirect()->back()->with('error', 'Failed to delete admin');

        } catch (ClientException $e) {
            $status = $e->getResponse()->getStatusCode();
            $body   = (string) $e->getResponse()->getBody();

            Log::error("DSpace DELETE ERROR", [
                'status' => $status,
                'body'   => $body
            ]);

            // unauthorized
            if ($status == 401) {
                session()->flush();
                return redirect()->route('auth-login-basic')
                    ->with('error', 'Authentication expired. Please login again.');
            }

            return redirect()->back()->with('error', 'Failed to delete admin');
        } catch (\Exception $e) {
            return response()->json(['error' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }
}
