<?php

namespace App\Http\Controllers\pages;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use phpseclib3\Net\SFTP;
use phpseclib3\Net\SSH2;

class OrgUnitItemController extends Controller
{
    private $host;
    private $port;
    private $username;
    private $password;

    public function __construct()
    {
        $this->host = config('services.putty.host'); // Load from .env via config
        $this->port = config('services.putty.port');
        $this->username = config('services.putty.username');
        $this->password = config('services.putty.password');
    }
    // index function for collection to Item
    public function index($id, Request $request)
    {
        if (!session()->has('api_token')) {
            return redirect()->route('auth-login-basic');
        }

        $publicationsData = [];
        $years = [];

        // Call the new function
        $person = $this->getPersonsByOrgUnitId($id);
        $personcount = $person->count();

        // getting data of publication 
        $publications = $this->getPublicationByOrgUnitId($id);
        $publicationcount = $publications->count();


        $totalcount = $personcount + $publicationcount;
        // dd($publication);
        $title = DB::table('metadatafieldregistry')
            ->where('element', 'title')
            ->whereNull('qualifier')
            ->where('metadata_schema_id', 1)
            ->value('metadata_field_id');

        $name = DB::table('metadatavalue')
            ->where('dspace_object_id', $id)
            ->where('metadata_field_id', $title)
            ->value('text_value');

        $publicationsData = [];
        $years = [];

        if (!empty($publications) && $publicationcount > 0) {
            foreach ($publications as $publication) {
                $metadataArray = json_decode($publication->metadata, true);

                $dateIssued = $this->getMetadataValue($metadataArray, 'date_issued');
                if (!empty($dateIssued)) {
                    $year = \Carbon\Carbon::parse($dateIssued)->format('Y');
                    $years[] = (int) $year;
                }

                // Extract metadata
                $title = $this->getMetadataValue($metadataArray, 'title');
                $publication_type = $this->getMetadataValue($metadataArray, 'publication_type');
                $authors = $this->getMetadataValue($metadataArray, 'authors');
                $journal = $this->getMetadataValue($metadataArray, 'journal');
                $doi = $this->getMetadataValue($metadataArray, 'doi');
                $date_issued = $this->getMetadataValue($metadataArray, 'date_issued');
                $volume = $this->getMetadataValue($metadataArray, 'volume');
                $abstract = $this->getMetadataValue($metadataArray, 'abstract');
                $eid = $this->getMetadataValue($metadataArray, 'eid');
                $scopus_citation = $this->getMetadataValue($metadataArray, 'scopus_citation') ?: 0;
                $scopus_author_id = $this->getMetadataValue($metadataArray, 'scopus_author_id');
                $isbn = $this->getMetadataValue($metadataArray, 'isbn');
                $issn = $this->getMetadataValue($metadataArray, 'issn');
                $upurl = $this->getMetadataValue($metadataArray, 'upurl');
                $sherpa_url = $this->getMetadataValue($metadataArray, 'shepa_url');
                $unpaywall = $this->getMetadataValue($metadataArray, 'unpaywall');
                $crossref_citation = $this->getMetadataValue($metadataArray, 'crossref_citation') ?: 0;

                $publicationsData[] = [
                    'uuid' => $publication->dspace_object_id,
                    'title' => $title,
                    'publication_type' => $publication_type,
                    'abstract' => $abstract,
                    'journal' => $journal,
                    'date_issued' => $date_issued,
                    'isbn' => $isbn,
                    'issn' => $issn,
                    'doi' => $doi,
                    'upurl' => $upurl,
                    'sherpa_url' => $sherpa_url,
                    'authors' => $authors,
                    'scopus_author_id' => $scopus_author_id,
                    'volume' => $volume,
                    'eid' => $eid,
                    'unpaywall' => $unpaywall,
                    'scopus_citation' => $scopus_citation,
                    'crossref_citation' => $crossref_citation,
                ];
            }

            // Filters
            $searchQuery = $request->input('search');
            if ($searchQuery) {
                $publicationsData = array_filter($publicationsData, function ($item) use ($searchQuery) {
                    return stripos($item['title'] ?? '', $searchQuery) !== false ||
                        stripos($item['authors'] ?? '', $searchQuery) !== false ||
                        stripos($item['journal'] ?? '', $searchQuery) !== false ||
                        stripos($item['publication_type'] ?? '', $searchQuery) !== false ||
                        stripos($item['scopus_citation'] ?? '', $searchQuery) !== false ||
                        stripos($item['crossref_citation'] ?? '', $searchQuery) !== false ||
                        stripos($item['date_issued'] ?? '', $searchQuery) !== false ||
                        stripos($item['doi'] ?? '', $searchQuery) !== false;
                });
            }

            $typeFilter = $request->input('type');
            if ($typeFilter && $typeFilter !== 'all') {
                $publicationsData = array_filter($publicationsData, function ($item) use ($typeFilter) {
                    return isset($item['publication_type']) &&
                        strtolower($item['publication_type']) === strtolower($typeFilter);
                });
            }

            $fromYear = $request->input('from_year');
            $toYear = $request->input('to_year');
            if ($fromYear || $toYear) {
                $publicationsData = array_filter($publicationsData, function ($item) use ($fromYear, $toYear) {
                    if (empty($item['date_issued'])) return false;
                    $year = \Carbon\Carbon::parse($item['date_issued'])->format('Y');
                    return (!$fromYear || $year >= $fromYear) && (!$toYear || $year <= $toYear);
                });
            }

            // Pagination
            $page = $request->input('page', 1);
            $perPage = $request->input('perPage', 10);
            $offset = ($page - 1) * $perPage;
            $publicationsData = array_values($publicationsData);
            $currentPageItems = array_slice($publicationsData, $offset, $perPage);

            $publications = new LengthAwarePaginator(
                $currentPageItems,
                count($publicationsData),
                $perPage,
                $page,
                [
                    'path' =>  url()->current(), // <-- Fixes incorrect pagination base path
                'query' => $request->query()]
            );

            $minYear = !empty($years) ? min($years) : null;
            $maxYear = !empty($years) ? max($years) : null;
        } else {
            // Set defaults if no publications
            $searchQuery = $request->input('search');
            $typeFilter = $request->input('type');
            $fromYear = $request->input('from_year');
            $toYear = $request->input('to_year');
            $publications = new LengthAwarePaginator([], 0, 10);
            $minYear = null;
            $maxYear = null;
        }

        return view('content.pages.orgunititem', compact(
            'minYear',
            'maxYear',
            'fromYear',
            'toYear',
            'searchQuery',
            'typeFilter',
            'person',
            'personcount',
            'publicationcount',
            'id',
            'name',
            'publications'
        ));
    }

    // for getting MetadataValue
    private function getMetadataValue($metadataArray, $fieldName)
    {
        return $metadataArray[$fieldName] ?? '';
    }

    // Get person data
    private function getPersonsByOrgUnitId($id)
    {
        // First, fetch all required metadata field IDs (you can further optimize this by caching or centralizing)
        $title = DB::table('metadatafieldregistry')->where('element', 'title')->whereNull('qualifier')->where('metadata_schema_id', 1)->value('metadata_field_id');
        $affiliationame = DB::table('metadatafieldregistry')->where('element', 'affiliation')->where('qualifier', 'name')->value('metadata_field_id');
        $email = DB::table('metadatafieldregistry')->where('element', 'email')->whereNull('qualifier')->value('metadata_field_id');
        $scopusauthorid = DB::table('metadatafieldregistry')->where('element', 'identifier')->where('qualifier', 'scopus-author-id')->value('metadata_field_id');
        $entitytype = DB::table('metadatafieldregistry')->where('element', 'entity')->where('qualifier', 'type')->value('metadata_field_id');
        $gender = DB::table('metadatafieldregistry')->where('element', 'person')->where('qualifier', 'gender')->value('metadata_field_id');
        $affiliationid = DB::table('metadatafieldregistry')->where('element', 'affiliation')->where('qualifier', 'id')->value('metadata_field_id');
        $jobtitle = DB::table('metadatafieldregistry')->where('element', 'jobTitle')->whereNull('qualifier')->value('metadata_field_id');
        $gsid = DB::table('metadatafieldregistry')->where('element', 'identifier')->where('qualifier', 'gsid')->value('metadata_field_id');
        $orcid = DB::table('metadatafieldregistry')->where('element', 'identifier')->where('qualifier', 'orcid')->value('metadata_field_id');
        $rid = DB::table('metadatafieldregistry')->where('element', 'identifier')->where('qualifier', 'rid')->value('metadata_field_id');

        $metadataFieldIds = [$title, $affiliationame, $email, $scopusauthorid, $entitytype, $gender, $affiliationid, $jobtitle, $gsid, $orcid, $rid];

        $person = DB::table('metadatavalue as mv_outer')
            ->select(
                'mv_outer.dspace_object_id',
                DB::raw("MAX(CASE WHEN mv_outer.metadata_field_id = $title THEN COALESCE(mv_outer.text_value, '') ELSE '' END) as preferred_name"),
                DB::raw("MAX(CASE WHEN mv_outer.metadata_field_id = $affiliationame THEN COALESCE(mv_outer.text_value, '') ELSE '' END) as main_affiliation"),
                DB::raw("MAX(CASE WHEN mv_outer.metadata_field_id = $email THEN COALESCE(mv_outer.text_value, '') ELSE '' END) as email"),
                DB::raw("MAX(CASE WHEN mv_outer.metadata_field_id = $scopusauthorid THEN COALESCE(mv_outer.text_value, '') ELSE '' END) as scopus_author_id"),
                DB::raw("MAX(CASE WHEN mv_outer.metadata_field_id = $entitytype THEN COALESCE(mv_outer.text_value, '') ELSE '' END) as entity_type"),
                DB::raw("MAX(CASE WHEN mv_outer.metadata_field_id = $gender THEN COALESCE(mv_outer.text_value, '') ELSE '' END) as gender"),
                DB::raw("MAX(CASE WHEN mv_outer.metadata_field_id = $affiliationid THEN COALESCE(mv_outer.text_value, '') ELSE '' END) as affiliation_id"),
                DB::raw("MAX(CASE WHEN mv_outer.metadata_field_id = $jobtitle THEN COALESCE(mv_outer.text_value, '') ELSE '' END) as job_title"),
                DB::raw("MAX(CASE WHEN mv_outer.metadata_field_id = $gsid THEN COALESCE(mv_outer.text_value, '') ELSE '' END) as gsid"),
                DB::raw("MAX(CASE WHEN mv_outer.metadata_field_id = $orcid THEN COALESCE(mv_outer.text_value, '') ELSE '' END) as orcid"),
                DB::raw("MAX(CASE WHEN mv_outer.metadata_field_id = $rid THEN COALESCE(mv_outer.text_value, '') ELSE '' END) as rid"),
                DB::raw("MAX(CASE WHEN mv_outer.metadata_field_id = $affiliationame THEN 
                (SELECT mv_inner.authority 
                FROM metadatavalue as mv_inner 
                WHERE mv_inner.metadata_field_id = $affiliationame
                AND mv_inner.dspace_object_id = mv_outer.dspace_object_id 
                LIMIT 1) 
                ELSE '' END) as department")
            )
            ->whereIn('mv_outer.dspace_object_id', function ($query) use ($id, $affiliationame) {
                $query->select('dspace_object_id')
                    ->from('metadatavalue')
                    ->whereIn('dspace_object_id', function ($subQuery) use ($id) {
                        $subQuery->select('dspace_object_id')
                            ->from('metadatavalue')
                            ->where('authority', $id);
                    })
                    ->where('metadata_field_id', $affiliationame);
            })
            ->whereIn('mv_outer.metadata_field_id', $metadataFieldIds)
            ->whereExists(function ($query) use ($entitytype) {
                $query->select(DB::raw(1))
                    ->from('metadatavalue as mv_check')
                    ->whereColumn('mv_check.dspace_object_id', 'mv_outer.dspace_object_id')
                    ->where('mv_check.metadata_field_id', $entitytype)
                    ->where('mv_check.text_value', 'Person');
            })
            ->groupBy('mv_outer.dspace_object_id')
            ->get();

        return $person;
    }

    // Get publicatcion data based on orgunit id
    private function getPublicationByOrgUnitId($id)
    {
        // Get all metadata field IDs
        $title = DB::table('metadatafieldregistry')
            ->where('element', 'title')
            ->whereNull('qualifier')
            ->where('metadata_schema_id', 1)
            ->value('metadata_field_id');
        $publication_type = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'subtypeDescription')
            ->where('metadata_schema_id', 1)
            ->value('metadata_field_id');
        $journal =  DB::table('metadatafieldregistry')
            ->where('element', 'relation')
            ->where('qualifier', 'ispartof')
            ->where('metadata_schema_id', '1')
            ->value('metadata_field_id');
        $isbn = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'isbn')
            ->value('metadata_field_id');
        $issn = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'issn')
            ->value('metadata_field_id');
        $date_issued = DB::table('metadatafieldregistry')
            ->where('element', 'date')
            ->where('qualifier', 'issued')
            ->value('metadata_field_id');
        $sherpa_url = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'sherpaUrl')
            ->value('metadata_field_id');
        $unpaywall = DB::table('metadatafieldregistry')
            ->where('element', 'venue')
            ->where('qualifier', 'unpaywall')
            ->value('metadata_field_id');
        $upurl = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'upurl')
            ->value('metadata_field_id');
        $crossref_citation = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'crossref_citation')
            ->value('metadata_field_id');
        $scopus_citation =  DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'citedby')
            ->value('metadata_field_id');
        $authors =  DB::table('metadatafieldregistry')
            ->where('element', 'contributor')
            ->where('qualifier', 'author')
            ->value('metadata_field_id');
        $scopus_author_id = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'scopus-author-id')
            ->value('metadata_field_id');
        $doi = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'doi')
            ->value('metadata_field_id');
        $eid = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'scopus')
            ->value('metadata_field_id');
        $volume = DB::table('metadatafieldregistry')
            ->where('element', 'citation')
            ->where('qualifier', 'volume')
            ->value('metadata_field_id');
        $abstract = DB::table('metadatafieldregistry')
            ->where('element', 'description')
            ->where('qualifier', 'abstract')
            ->value('metadata_field_id');
        $departmentId = DB::table('metadatafieldregistry')
            ->where('element', 'virtual')
            ->where('qualifier', 'department')
            ->value('metadata_field_id');

        // Using your exact query to get filtered objects
        $filteredObjects = DB::table('metadatavalue')
            ->select('dspace_object_id')
            ->where('authority', $id)
            ->where('metadata_field_id', $departmentId)
            ->distinct()
            ->pluck('dspace_object_id')
            ->toArray();

        // Aggregate metadata for the filtered objects
        $aggregatedMetadata = DB::table('metadatavalue as mv')
            ->select(
                'mv.dspace_object_id',
                DB::raw("
                CASE 
                    WHEN mv.metadata_field_id = $title THEN 'title'
                    WHEN mv.metadata_field_id = $publication_type THEN 'publication_type'
                    WHEN mv.metadata_field_id = $date_issued THEN 'date_issued'
                    WHEN mv.metadata_field_id = $isbn THEN 'isbn'
                    WHEN mv.metadata_field_id = $issn THEN 'issn'
                    WHEN mv.metadata_field_id = $upurl THEN 'upurl'
                    WHEN mv.metadata_field_id = $authors THEN 'authors'
                    WHEN mv.metadata_field_id = $doi THEN 'doi'
                    WHEN mv.metadata_field_id = $eid THEN 'eid'
                    WHEN mv.metadata_field_id = $journal THEN 'journal'
                    WHEN mv.metadata_field_id = $scopus_author_id THEN 'scopus_author_id'
                    WHEN mv.metadata_field_id = $scopus_citation THEN 'scopus_citation'
                    WHEN mv.metadata_field_id = $volume THEN 'volume'
                    WHEN mv.metadata_field_id = $unpaywall THEN 'unpaywall'
                    WHEN mv.metadata_field_id = $crossref_citation THEN 'crossref_citation'
                    WHEN mv.metadata_field_id = $sherpa_url THEN 'sherpa_url'
                    WHEN mv.metadata_field_id = $abstract THEN 'abstract'
                    ELSE 'unknown_field_' || mv.metadata_field_id
                    END AS field_name,
                    CASE    
                    WHEN mv.metadata_field_id IN ($authors, $scopus_author_id) THEN STRING_AGG(mv.text_value, ' || ') 
                    ELSE TRIM(BOTH '[]\"' FROM JSON_AGG(mv.text_value)::TEXT) 
                    END AS text_values
                    ")
            )
            ->whereIn('mv.dspace_object_id', $filteredObjects)
            ->whereIn('mv.metadata_field_id', [$title, $date_issued, $isbn, $issn, $upurl, $authors, $doi, $eid, $journal, $scopus_author_id, $scopus_citation, $volume, $unpaywall, $crossref_citation, $sherpa_url, $abstract, $publication_type])
            ->groupBy('mv.dspace_object_id', 'mv.metadata_field_id');

        // Create structured publications data
        $publications = DB::table(DB::raw("({$aggregatedMetadata->toSql()}) as am"))
            ->mergeBindings($aggregatedMetadata)
            ->select(
                'am.dspace_object_id',
                DB::raw("JSON_OBJECT_AGG(am.field_name, am.text_values) AS metadata")
            )
            ->groupBy('am.dspace_object_id')
            ->get();

        // Return the publications data directly
        return $publications;
    }

    //    It will export publication csv under perticular orgunit
    public function exportPublication(Request $request)
    {
        if (!session()->has('api_token')) {
            return redirect()->route('auth-login-basic');
        }

        try {

            // Taking name and id from blade file
            $name = $request->input('orgname');
            $id = $request->input('orgid');

            // Publication data 
            $title = DB::table('metadatafieldregistry')
                ->where('element', 'title')
                ->whereNull('qualifier')
                ->where('metadata_schema_id', 1)
                ->value('metadata_field_id');
            // Entity type
            $entitytype = DB::table('metadatafieldregistry')->where('element', 'entity')->where('qualifier', 'type')->value('metadata_field_id');

            // handle id
            $handleMetadataId = DB::table('metadatafieldregistry')
                ->where('element', 'identifier')
                ->where('qualifier', 'uri')
                ->value('metadata_field_id');

            // submission defination
            $subDefination = DB::table('metadatafieldregistry')
                ->where('element', 'submission')
                ->where('qualifier', 'definition')
                ->value('metadata_field_id');

            $allPunhandleId = DB::table('public.metadatavalue')
                ->select(DB::raw("SUBSTRING(text_value FROM 'handle/([^/]+/[0-9]+)') AS handle_id"))
                ->where('metadata_field_id', $handleMetadataId)
                ->where('dspace_object_id', function ($query) use ($subDefination) {
                    $query->select('dspace_object_id')
                        ->from('public.metadatavalue')
                        ->where('metadata_field_id', $subDefination)
                        ->where(function ($q) {
                            $q->where('text_value', 'publication')
                                ->orWhere('text_value', 'publication-edit');
                        })
                        ->limit(1);
                })
                ->value('handle_id');

            $host = $this->host;
            $port = $this->port;
            $username = $this->username;
            $password = $this->password;

            $remoteFullCSV = '/tmp/full_metadata.csv';
            $localFullCSV = storage_path('app/full_metadata.csv');
            $filteredCSV = storage_path('app/filtered_metadata.csv');

            // Keyword to filter (e.g., your OrgUnit/Department string)
            $filterKeyword = "{$name}::{$id}::600";

            // dd($filterKeyword);

            // Step 1: SSH into server and export all publication metadata
            $ssh = new SSH2($host, $port);
            if (!$ssh->login($username, $password)) {
                throw new \Exception("SSH login failed.");
            }

            $exportCommand = "echo " . escapeshellarg($password) . " | sudo -S /dspace/bin/dspace metadata-export -f {$remoteFullCSV} -i {$allPunhandleId}";
            // dd($exportCommand);
            $ssh->exec($exportCommand);

            // Step 2: Download full metadata CSV
            $sftp = new SFTP($host, $port);
            if (!$sftp->login($username, $password)) {
                throw new \Exception("SFTP login failed.");
            }

            if (!$sftp->get($remoteFullCSV, $localFullCSV)) {
                throw new \Exception("Failed to download full CSV.");
            }

            // Step 3: Open CSV and filter
            $rows = array_map('str_getcsv', file($localFullCSV));
            if (empty($rows)) {
                throw new \Exception("CSV file is empty.");
            }

            $header = $rows[0];
            $filteredRows = [$header];

            foreach ($rows as $i => $row) {
                if ($i === 0) continue; // Skip header
                if (collect($row)->contains(fn($cell) => str_contains($cell, $filterKeyword))) {
                    $filteredRows[] = $row;
                }
            }

            // Step 4: Write filtered CSV
            $fp = fopen($filteredCSV, 'w');
            foreach ($filteredRows as $row) {
                fputcsv($fp, $row);
            }
            fclose($fp);

          
            // Delete the original unfiltered CSV
            if (file_exists($localFullCSV)) {
                unlink($localFullCSV);
            }


            // Step 5: Return filtered CSV as download
            return response()->download($filteredCSV, "metadata_{$id}.csv", [
                'Content-Type' => 'text/csv',
            ])->deleteFileAfterSend(true);
        } catch (\Exception $e) {
            Log::error('Metadata Filter Export Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Export failed: ' . $e->getMessage());
        }
    }
}
