<?php

namespace App\Http\Controllers\pages;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;


class PersonStatisticsController extends Controller
{
    protected $titleFieldId;
    protected $entityTypeId;
    protected $citedById;
    protected $unpaywallId;
    protected $scopusId;
    protected $wosId;
    protected $crossrefCitationId;
    protected $subjectScopusId;
    protected $subjectWosId;
    protected $typeId;
    protected $issueDateId;
    protected $authorContriID;


    public function __construct()
    {

        $this->titleFieldId = DB::table('metadatafieldregistry')
            ->where('element', 'title')
            ->whereNull('qualifier')
            ->where('metadata_schema_id', 1)
            ->value('metadata_field_id');

        $this->entityTypeId = DB::table('metadatafieldregistry')
            ->where('element', 'entity')
            ->where('qualifier', 'type')
            ->value('metadata_field_id');

        $this->citedById = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'citedby')
            ->value('metadata_field_id');

        $this->unpaywallId = DB::table('metadatafieldregistry')
            ->where('element', 'venue')
            ->where('qualifier', 'unpaywall')
            ->value('metadata_field_id');

        $this->scopusId = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'scopus')
            ->value('metadata_field_id');

        $this->wosId = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'wos')
            ->value('metadata_field_id');

        $this->crossrefCitationId = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'crossref_citation')
            ->value('metadata_field_id');

        $this->subjectScopusId = DB::table('metadatafieldregistry')
            ->where('element', 'subject_scopus')
            // ->where('qualifier', 'scopus')
            ->value('metadata_field_id');

        $this->subjectWosId = DB::table('metadatafieldregistry')
            ->where('element', 'subject_wos')
            // ->where('qualifier', 'scopus')
            ->value('metadata_field_id');

        $this->typeId = DB::table('metadatafieldregistry')
            ->where('element', 'type')
            ->where('metadata_schema_id', function ($query) {
                $query->select('metadata_schema_id')
                    ->from('metadataschemaregistry')
                    ->where('short_id', 'dc')
                    ->limit(1);
            })
            ->value('metadata_field_id');

        $this->issueDateId = DB::table('metadatafieldregistry')
            ->where('element', 'date')
            ->where('qualifier', 'issued')
            ->value('metadata_field_id');

        $this->authorContriID = DB::table('metadatafieldregistry')
            ->where('element', 'contributor')
            ->where('qualifier', 'author')
            ->value('metadata_field_id');
    }

    //index method
    public function index()
    {
        return view('content.pages.facultypublications-count');
    }

    /**
     * Return Persons Statestic data for DataTable (AJAX)
     */
    public function getPersonStatestic(Request $request)
    {
        $yearFrom = $request->year_from;
        $yearTo   = $request->year_to;
        // -----------------------------------------
        // CASE 1: NO FILTER → Load from JSON
        // -----------------------------------------
        if (empty($yearFrom) || empty($yearTo)) {
            $jsonPath = storage_path('app/json/person_statestic.json');
            
            if (!file_exists($jsonPath)) {
                return response()->json(['data' => []]);
            }
            
            $json = json_decode(file_get_contents($jsonPath), true);
            return response()->json(['data' => $json]);
        }
        
        // -----------------------------------------
        // CASE 2: FILTER APPLIED → Use CACHE + DB
        // -----------------------------------------
        
        $cacheKey = "person_statestic_{$yearFrom}_{$yearTo}";
        // dd($yearFrom,$yearTo);

        $data = Cache::remember($cacheKey, now()->addHours(3), function () use ($yearFrom, $yearTo) {
            return $this->runFilterQuery($yearFrom, $yearTo);
        });

        return response()->json(['data' => $data]);
    }


    public function updateData()
    {
        try {
            // 1️⃣ Run heavy query
            $data = $this->getHeavyQuery();

            // 2️⃣ Prepare JSON
            $json = json_encode($data, JSON_PRETTY_PRINT);

            if ($json === false) {
                throw new \Exception("JSON encoding failed");
            }

            // 3️⃣ Save safely (write to temp, then rename)
            $tempPath = storage_path('app/json/person_count_temp.json');
            $finalPath = storage_path('app/json/person_statestic.json');

            file_put_contents($tempPath, $json);

            // Atomic replace (prevents corrupted file)
            rename($tempPath, $finalPath);

            // 4️⃣ Flash success message
            session()->flash('success', 'Person statistics updated successfully!');

            return response()->json([
                'success' => true,
                'message' => session('success')
            ]);
        } catch (\Throwable $e) {

            // 5️⃣ Log the error for debugging
            Log::error('Person statistics update failed: ' . $e->getMessage());

            // 6️⃣ Return JSON error (so JavaScript can show message)
            return response()->json([
                'success' => false,
                'error'   => 'Failed to update Persons statistics. Please try again.',
                'details' => $e->getMessage()  // Remove in production if needed
            ], 500);
        }
    }

    private function getHeavyQuery()
    {
        $titleFieldId = $this->titleFieldId;
        $entityTypeId = $this->entityTypeId;
        $citedById = $this->citedById;
        $unpaywallId = $this->unpaywallId;
        $scopusId = $this->scopusId;
        $wosId = $this->wosId;
        $crossrefCitationId = $this->crossrefCitationId;
        $subjectScopusId = $this->subjectScopusId;
        $subjectWosId = $this->subjectWosId;
        $typeId = $this->typeId;
        $authorContriID = $this->authorContriID;

        return DB::table('collection2item as p_item')
            ->selectRaw("
                p_item.item_id AS person_id,
                p_name.text_value AS person_name,

                COUNT(DISTINCT pub_author.dspace_object_id) AS total_publications,

                SUM(COALESCE(pub_cite.citation_count, 0)) AS total_publication_citations,

                COUNT(DISTINCT CASE
                    WHEN oa_meta.text_value IS NOT NULL
                    AND oa_meta.text_value IN ('green','gold','hybrid','bronze')
                    THEN pub_author.dspace_object_id END
                ) AS open_access_publications,

                SUM(CASE
                    WHEN oa_meta.text_value IS NOT NULL
                    AND oa_meta.text_value IN ('green','gold','hybrid','bronze')
                    THEN COALESCE(pub_cite.citation_count, 0)
                END) AS open_access_citations,

                COUNT(DISTINCT CASE
                    WHEN scopus_meta.dspace_object_id IS NOT NULL
                    THEN pub_author.dspace_object_id END
                ) AS scopus_publications,

                SUM(CASE WHEN scopus_meta.dspace_object_id IS NOT NULL
                    THEN COALESCE(pub_cite.citation_count, 0)
                END) AS scopus_citations,

                COUNT(DISTINCT CASE
                    WHEN wos_meta.dspace_object_id IS NOT NULL
                    THEN pub_author.dspace_object_id END
                ) AS wos_publications,

                SUM(CASE WHEN wos_meta.dspace_object_id IS NOT NULL
                    THEN COALESCE(wos_cite_meta.text_value::INTEGER, 0)
                END) AS wos_citations,

                COUNT(DISTINCT CASE
                    WHEN scopus_meta.dspace_object_id IS NULL
                    AND wos_meta.dspace_object_id IS NULL
                    THEN pub_author.dspace_object_id END
                ) AS not_indexed_publications,

                SUM(CASE
                    WHEN scopus_meta.dspace_object_id IS NULL
                    AND wos_meta.dspace_object_id IS NULL
                    THEN COALESCE(pub_cite.citation_count, 0)
                END) AS not_indexed_citations,

                COUNT(DISTINCT CASE
                    WHEN scopus_eng_meta.text_value = 'ENG'
                    THEN pub_author.dspace_object_id END
                ) AS scopus_eng_publications,

                SUM(CASE
                    WHEN scopus_eng_meta.text_value = 'ENG'
                    THEN COALESCE(pub_cite.citation_count, 0)
                END) AS scopus_eng_citations,

                COUNT(DISTINCT CASE
                    WHEN wos_eng_meta.text_value = 'ENG'
                    THEN pub_author.dspace_object_id END
                ) AS wos_eng_publications,

                SUM(CASE
                    WHEN wos_eng_meta.text_value = 'ENG'
                    THEN COALESCE(wos_cite_meta.text_value::INTEGER, 0)
                END) AS wos_eng_citations
                ")

            /* Person type */
            ->join('metadatavalue as p_type', function ($join) use ($entityTypeId) {
                $join->on('p_type.dspace_object_id', '=', 'p_item.item_id')
                    ->where('p_type.metadata_field_id', $entityTypeId)
                    ->where('p_type.text_value', 'Person');
            })

            /* Person name */
            ->join('metadatavalue as p_name', function ($join) use ($titleFieldId) {
                $join->on('p_name.dspace_object_id', '=', 'p_item.item_id')
                    ->where('p_name.metadata_field_id', $titleFieldId);
            })

            /* Author relations */
            ->leftJoin('metadatavalue as pub_author', function ($join) use ($authorContriID) {
                $join->on('pub_author.metadata_field_id', '=', DB::raw($authorContriID))
                    ->on('pub_author.authority', '=', DB::raw("p_item.item_id::text"));
            })

            /* Publication type */
            ->leftJoin('metadatavalue as pub_type', function ($join) use ($entityTypeId) {
                $join->on('pub_type.dspace_object_id', '=', 'pub_author.dspace_object_id')
                    ->where('pub_type.metadata_field_id', $entityTypeId)
                    ->where('pub_type.text_value', 'Publication');
            })

            /* Citations (subquery) */
            ->leftJoin(
                DB::raw("(SELECT dspace_object_id, MAX(text_value::INTEGER) AS citation_count
                         FROM metadatavalue
                         WHERE metadata_field_id = $citedById
                         GROUP BY dspace_object_id) as pub_cite"),
                'pub_cite.dspace_object_id',
                '=',
                'pub_author.dspace_object_id'
            )

            /* OA */
            ->leftJoin('metadatavalue as oa_meta', function ($join) use ($unpaywallId) {
                $join->on('oa_meta.dspace_object_id', '=', 'pub_author.dspace_object_id')
                    ->where('oa_meta.metadata_field_id', $unpaywallId);
            })

            /* Scopus */
            ->leftJoin('metadatavalue as scopus_meta', function ($join) use ($scopusId) {
                $join->on('scopus_meta.dspace_object_id', '=', 'pub_author.dspace_object_id')
                    ->where('scopus_meta.metadata_field_id', $scopusId);
            })

            /* WoS */
            ->leftJoin('metadatavalue as wos_meta', function ($join) use ($wosId) {
                $join->on('wos_meta.dspace_object_id', '=', 'pub_author.dspace_object_id')
                    ->where('wos_meta.metadata_field_id', $wosId);
            })

            /* WOS citations */
            ->leftJoin('metadatavalue as wos_cite_meta', function ($join) use ($crossrefCitationId) {
                $join->on('wos_cite_meta.dspace_object_id', '=', 'pub_author.dspace_object_id')
                    ->where('wos_cite_meta.metadata_field_id', $crossrefCitationId);
            })

            /* Scopus ENG */
            ->leftJoin('metadatavalue as scopus_eng_meta', function ($join) use ($subjectScopusId) {
                $join->on('scopus_eng_meta.dspace_object_id', '=', 'pub_author.dspace_object_id')
                    ->where('scopus_eng_meta.metadata_field_id', $subjectScopusId);
            })

            /* WOS ENG */
            ->leftJoin('metadatavalue as wos_eng_meta', function ($join) use ($subjectWosId) {
                $join->on('wos_eng_meta.dspace_object_id', '=', 'pub_author.dspace_object_id')
                    ->where('wos_eng_meta.metadata_field_id', $subjectWosId);
            })

            ->whereNotNull('pub_type.dspace_object_id')   // only publications
            ->groupBy('p_item.item_id', 'p_name.text_value')
            ->orderBy('total_publications', 'DESC')
            ->get();
    }

    private function runFilterQuery($yearFrom, $yearTo)
    {

        $yearFrom = $yearFrom;
        $yearTo   = $yearTo;
        $titleFieldId = $this->titleFieldId;
        $entityTypeId = $this->entityTypeId;
        $citedById = $this->citedById;
        $unpaywallId = $this->unpaywallId;
        $scopusId = $this->scopusId;
        $wosId = $this->wosId;
        $crossrefCitationId = $this->crossrefCitationId;
        $subjectScopusId = $this->subjectScopusId;
        $subjectWosId = $this->subjectWosId;
        $typeId = $this->typeId;
        $issueDateId = $this->issueDateId;
        $authorContriID = $this->authorContriID;
        return DB::table('collection2item as p_item')

            /* ===== Person Type ===== */
            ->join('metadatavalue as p_type', function ($join) use ($entityTypeId) {
                $join->on('p_type.dspace_object_id', '=', 'p_item.item_id')
                    ->where('p_type.metadata_field_id', $entityTypeId)
                    ->where('p_type.text_value', 'Person');
            })

            /* ===== Person Name ===== */
            ->join('metadatavalue as p_name', function ($join) use ($titleFieldId) {
                $join->on('p_name.dspace_object_id', '=', 'p_item.item_id')
                    ->where('p_name.metadata_field_id', $titleFieldId);
            })

            /* ===== Author → Publication ===== */
            ->leftJoin('metadatavalue as pub_author', function ($join) use ($authorContriID) {
                $join->on('pub_author.metadata_field_id', '=', DB::raw($authorContriID))
                    ->on('pub_author.authority', '=', DB::raw("p_item.item_id::text"));
            })

            /* Ensure it is a Publication */
            ->leftJoin('metadatavalue as pub_type', function ($join) use ($entityTypeId) {
                $join->on('pub_type.dspace_object_id', '=', 'pub_author.dspace_object_id')
                    ->where('pub_type.metadata_field_id', $entityTypeId)
                    ->where('pub_type.text_value', 'Publication');
            })

            /* ===== ISSUE YEAR METADATA ===== */
            ->leftJoin('metadatavalue as issue_date_meta', function ($join) use ($issueDateId) {
                $join->on('issue_date_meta.dspace_object_id', '=', 'pub_author.dspace_object_id')
                    ->where('issue_date_meta.metadata_field_id', $issueDateId); // <-- Issue Date Field
            })

            /* ===== Citation Subquery ===== */
            ->leftJoin(
                DB::raw("(SELECT dspace_object_id, MAX(text_value::INTEGER) AS citation_count
                  FROM metadatavalue
                  WHERE metadata_field_id = $citedById
                  GROUP BY dspace_object_id) as pub_cite"),
                'pub_cite.dspace_object_id',
                '=',
                'pub_author.dspace_object_id'
            )

            /* OA */
            ->leftJoin('metadatavalue as oa_meta', function ($join) use ($unpaywallId) {
                $join->on('oa_meta.dspace_object_id', '=', 'pub_author.dspace_object_id')
                    ->where('oa_meta.metadata_field_id', $unpaywallId);
            })

            /* Scopus */
            ->leftJoin('metadatavalue as scopus_meta', function ($join) use ($scopusId) {
                $join->on('scopus_meta.dspace_object_id', '=', 'pub_author.dspace_object_id')
                    ->where('scopus_meta.metadata_field_id', $scopusId);
            })

            /* WoS */
            ->leftJoin('metadatavalue as wos_meta', function ($join) use ($wosId) {
                $join->on('wos_meta.dspace_object_id', '=', 'pub_author.dspace_object_id')
                    ->where('wos_meta.metadata_field_id', $wosId);
            })

            /* WoS Citations */
            ->leftJoin('metadatavalue as wos_cite_meta', function ($join) use ($crossrefCitationId) {
                $join->on('wos_cite_meta.dspace_object_id', '=', 'pub_author.dspace_object_id')
                    ->where('wos_cite_meta.metadata_field_id', $crossrefCitationId);
            })

            /* Scopus ENG */
            ->leftJoin('metadatavalue as scopus_eng_meta', function ($join) use ($subjectScopusId) {
                $join->on('scopus_eng_meta.dspace_object_id', '=', 'pub_author.dspace_object_id')
                    ->where('scopus_eng_meta.metadata_field_id', $subjectScopusId);
            })

            /* WoS ENG */
            ->leftJoin('metadatavalue as wos_eng_meta', function ($join) use ($subjectWosId) {
                $join->on('wos_eng_meta.dspace_object_id', '=', 'pub_author.dspace_object_id')
                    ->where('wos_eng_meta.metadata_field_id', $subjectWosId);
            })


            /* ===== WHERE CLAUSE ===== */
            ->whereNotNull('pub_type.dspace_object_id')

            // year not null
            ->whereNotNull('issue_date_meta.text_value')

            // valid YYYY-MM-DD
            ->whereRaw("issue_date_meta.text_value ~ '^[0-9]{4}-[0-9]{2}-[0-9]{2}$'")

            // year filter
            ->whereBetween(DB::raw("LEFT(issue_date_meta.text_value, 4)::INTEGER"), [$yearFrom, $yearTo])


            /* ===== SELECT FIELDS ===== */
            ->select([
                'p_item.item_id as person_id',
                'p_name.text_value as person_name',

                DB::raw("COUNT(DISTINCT pub_author.dspace_object_id) AS total_publications"),

                DB::raw("SUM(COALESCE(pub_cite.citation_count, 0)) AS total_publication_citations"),

                DB::raw("COUNT(DISTINCT CASE WHEN oa_meta.text_value IN ('green','gold','hybrid','bronze')
                                     THEN pub_author.dspace_object_id END) AS open_access_publications"),

                DB::raw("SUM(CASE WHEN oa_meta.text_value IN ('green','gold','hybrid','bronze')
                          THEN COALESCE(pub_cite.citation_count, 0) END) AS open_access_citations"),

                DB::raw("COUNT(DISTINCT CASE WHEN scopus_meta.dspace_object_id IS NOT NULL
                                     THEN pub_author.dspace_object_id END) AS scopus_publications"),

                DB::raw("SUM(CASE WHEN scopus_meta.dspace_object_id IS NOT NULL
                          THEN COALESCE(pub_cite.citation_count, 0) END) AS scopus_citations"),

                DB::raw("COUNT(DISTINCT CASE WHEN wos_meta.dspace_object_id IS NOT NULL
                                     THEN pub_author.dspace_object_id END) AS wos_publications"),

                DB::raw("SUM(CASE WHEN wos_meta.dspace_object_id IS NOT NULL
                          THEN COALESCE(wos_cite_meta.text_value::INTEGER, 0) END) AS wos_citations"),

                DB::raw("COUNT(DISTINCT CASE WHEN scopus_meta.dspace_object_id IS NULL
                                     AND wos_meta.dspace_object_id IS NULL
                                     THEN pub_author.dspace_object_id END) AS not_indexed_publications"),

                DB::raw("SUM(CASE WHEN scopus_meta.dspace_object_id IS NULL
                          AND wos_meta.dspace_object_id IS NULL
                          THEN COALESCE(pub_cite.citation_count, 0) END) AS not_indexed_citations"),

                DB::raw("COUNT(DISTINCT CASE WHEN scopus_eng_meta.text_value = 'ENG'
                                     THEN pub_author.dspace_object_id END) AS scopus_eng_publications"),

                DB::raw("SUM(CASE WHEN scopus_eng_meta.text_value = 'ENG'
                          THEN COALESCE(pub_cite.citation_count, 0) END) AS scopus_eng_citations"),

                DB::raw("COUNT(DISTINCT CASE WHEN wos_eng_meta.text_value = 'ENG'
                                     THEN pub_author.dspace_object_id END) AS wos_eng_publications"),

                DB::raw("SUM(CASE WHEN wos_eng_meta.text_value = 'ENG'
                          THEN COALESCE(wos_cite_meta.text_value::INTEGER, 0) END) AS wos_eng_citations"),
            ])

            ->groupBy('p_item.item_id', 'p_name.text_value')
            ->orderBy('total_publications', 'DESC')
            ->get();
    }
}
