<?php

namespace App\Http\Controllers\pages;


use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;
use App\Services\DSpaceApiService;
use GuzzleHttp\Exception\ClientException;
use Illuminate\Support\Facades\Log;

class ProcessesController extends Controller
{

    protected $title;

    // Constructor 
    public function __construct()
    {
        $this->title = DB::table('metadatafieldregistry')
            ->where('element', 'title')
            ->whereNull('qualifier')
            ->where('metadata_schema_id', 1)
            ->value('metadata_field_id');
    }
    // index function
    public function index()
    {
        $filePath = storage_path('app/json/processes.json'); // Path to the JSON file
        // $filePath = public_path('json/processes.json'); // Path to the JSON file
        $data = [];

        // Check if the file exists
        if (file_exists($filePath)) {
            // Read the JSON file
            $fileContent = file_get_contents($filePath);

            // Decode the JSON content to an array
            $data = json_decode($fileContent, true);
        }

        // Pass the data to the view
        return view('content.pages.processes', ['processes' => $data]);
    }

    // Method to create JSON file
    public function createJsonFile()
    {

        $filePath = storage_path('app/json/processes.json');

        $query = DB::table('process as p')
            ->select([
                'p.process_id',
                'p.start_time',
                'p.finished_time',
                'p.creation_time',
                'p.script',
                'p.parameters',
                'p.status',
                DB::raw("
            CONCAT(
                COALESCE(mv_firstname.text_value, ''), ' ', 
                COALESCE(mv_lastname.text_value, '')
            ) AS name
        "),
                DB::raw("
            STRING_AGG(DISTINCT m.text_value, ';') AS file_name
        "),
                DB::raw("
            STRING_AGG(DISTINCT pb.bitstream_id::text, ';') AS file_bitstream_id
        ")
            ])
            ->leftJoin('process2bitstream as pb', 'p.process_id', '=', 'pb.process_id')
            ->leftJoin('metadatavalue as m', function ($join) {
                $join->on('m.dspace_object_id', '=', 'pb.bitstream_id')
                    ->where('m.metadata_field_id', $this->title);
            })
            ->leftJoin('metadatavalue as mv_firstname', function ($join) {
                $join->on('mv_firstname.dspace_object_id', '=', 'p.user_id')
                    ->where('mv_firstname.metadata_field_id', 1);
            })
            ->leftJoin('metadatavalue as mv_lastname', function ($join) {
                $join->on('mv_lastname.dspace_object_id', '=', 'p.user_id')
                    ->where('mv_lastname.metadata_field_id', 2);
            })
            ->whereNotNull('p.user_id')
            ->groupBy([
                'p.process_id',
                'p.start_time',
                'p.finished_time',
                'p.creation_time',
                'p.script',
                'p.status',
                'p.parameters',
                'mv_firstname.text_value',
                'mv_lastname.text_value'
            ])
            ->orderByDesc('p.process_id');

        $allData = $query->get()->toArray();
        file_put_contents($filePath, json_encode($allData, JSON_PRETTY_PRINT));
    }


    public function process(Request $request)
    {
        if ($request->isMethod('get')) {
            if (session()->has('api_token')) {
                $this->createJsonFile();
                session()->flash('success', 'Process Updated Successfully!');

                return response()->json(['success' => session('success')]);
            } else {
                return response()->json(['error' => 'Failed to Update Process'], 400);
            }
        }
    }

    // Delete function
    public function delete($id, DSpaceApiService $apiService)
    {
        try {
            $authorizationHeader = Session::get('api_token');

            $response = $apiService->delete('system/processes/' . $id, $authorizationHeader);

            if ($response->getStatusCode() === 204) {
                // Call the function to create the JSON file after successfully deleting the process
                $this->createJsonFile();

                // Redirect to the desired view with a success message
                return redirect()->route('processes.index')->with('success', 'Process deleted successfully');
            } else {
                // Redirect back with an error message or handle the error accordingly
                return redirect()->back()->with('error', 'Failed to delete process');
            }
        } catch (ClientException $e) {
            $status = $e->getResponse()->getStatusCode();
            $body   = (string) $e->getResponse()->getBody();

            Log::error("DSpace DELETE ERROR", [
                'status' => $status,
                'body'   => $body
            ]);

            // unauthorized
            if ($status == 401) {
                session()->flush();
                return redirect()->route('auth-login-basic')
                    ->with('error', 'Authentication expired. Please login again.');
            }

            return redirect()->back()->with('error', 'Failed to delete admin');
        } catch (\Exception $e) {
            return response()->json(['error' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }
}
