<?php

namespace App\Http\Controllers\pages;


use App\Http\Controllers\Controller;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use App\Services\DSpaceApiService;

class PublicationsController extends Controller
{
    protected $isbn;
    protected $issn;
    protected $date_issued;
    protected $sherpa_url;
    protected $unpaywall;
    protected $upurl;
    protected $crossref_citation;
    protected $authors;
    protected $scopus_author_id;
    protected $doi;
    protected $eid;
    protected $volume;
    protected $abstract;
    protected $publication_type;
    protected $journal;
    protected $title;


    public function __construct()
    {
        $this->title = DB::table('metadatafieldregistry')
            ->where('element', 'title')
            ->whereNull('qualifier')
            ->where('metadata_schema_id', 1)
            ->value('metadata_field_id');

        $this->publication_type = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'subtypeDescription')
            ->where('metadata_schema_id', 1)
            ->value('metadata_field_id');

        $this->journal =  DB::table('metadatafieldregistry')
            ->where('element', 'relation')
            ->where('qualifier', 'ispartof')
            ->where('metadata_schema_id', '1')
            ->value('metadata_field_id');

        $this->isbn = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'isbn')
            ->value('metadata_field_id');

        $this->issn = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'issn')
            ->value('metadata_field_id');

        $this->date_issued = DB::table('metadatafieldregistry')
            ->where('element', 'date')
            ->where('qualifier', 'issued')
            ->value('metadata_field_id');

        $this->sherpa_url = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'sherpaUrl')
            ->value('metadata_field_id');


        $this->unpaywall = DB::table('metadatafieldregistry')
            ->where('element', 'venue')
            ->where('qualifier', 'unpaywall')
            ->value('metadata_field_id');

        $this->upurl = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'upurl')
            ->value('metadata_field_id');

        $this->crossref_citation = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'crossref_citation')
            ->value('metadata_field_id');

        $this->authors =  DB::table('metadatafieldregistry')
            ->where('element', 'contributor')
            ->where('qualifier', 'author')
            ->value('metadata_field_id');

        $this->scopus_author_id = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'scopus-author-id')
            ->value('metadata_field_id');

        $this->doi = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'doi')
            ->value('metadata_field_id');

        $this->eid = DB::table('metadatafieldregistry')
            ->where('element', 'identifier')
            ->where('qualifier', 'scopus')
            ->value('metadata_field_id');

        $this->volume = DB::table('metadatafieldregistry')
            ->where('element', 'citation')
            ->where('qualifier', 'volume')
            ->value('metadata_field_id');

        $this->abstract = DB::table('metadatafieldregistry')
            ->where('element', 'description')
            ->where('qualifier', 'abstract')
            ->value('metadata_field_id');
    }
    //index function
    public function index(Request $request)
    {
        // $filePath = public_path('json/publicationlist.json');
        $filePath = storage_path('app/json/publicationlist.json');
        $publicationsRaw = [];
        $publicationsData = [];
        $totalCount = 0;

        if (file_exists($filePath)) {
            $fileContent = file_get_contents($filePath);
            $publicationsRaw = json_decode($fileContent, true);

            if (!empty($publicationsRaw)) {
                foreach ($publicationsRaw as $publication) {
                    $metadataArray = json_decode($publication['metadata_details'], true);


                    // Year collection
                    $dateIssued = $this->getMetadataValue($metadataArray, 'date_issued');

                    // if (!empty($dateIssued)) {

                    //     $year = \Carbon\Carbon::parse($dateIssued)->format('Y');
                    //     $years[] = (int) $year;
                    // }
                    if (!empty($dateIssued)) {
                        // Remove quotes and extra spaces
                        $cleanDate = trim(str_replace('"', '', $dateIssued));

                        // If multiple dates separated by comma, take the last part
                        if (strpos($cleanDate, ',') !== false) {
                            $parts = explode(',', $cleanDate);
                            $cleanDate = trim(end($parts));
                        }

                        try {
                            $year = \Carbon\Carbon::parse($cleanDate)->format('Y');
                            $years[] = (int) $year;
                        } catch (\Exception $e) {
                            // Handle invalid dates gracefully
                            // For example, skip this entry
                        }
                    }


                    // if (!empty($dateIssued)) {
                    //     $dates[] = \Carbon\Carbon::parse($dateIssued)->format('Y-m-d'); // Store full date
                    // }

                    $title = $this->getMetadataValue($metadataArray, 'title');
                    $publication_type = $this->getMetadataValue($metadataArray, 'publication_type');
                    $authors = $this->getMetadataValue($metadataArray, 'authors');
                    $journal = $this->getMetadataValue($metadataArray, 'journal');
                    $doi = $this->getMetadataValue($metadataArray, 'doi');
                    $date_issued = $this->getMetadataValue($metadataArray, 'date_issued');
                    $volume = $this->getMetadataValue($metadataArray, 'volume');
                    $abstract = $this->getMetadataValue($metadataArray, 'abstract');
                    $eid = $this->getMetadataValue($metadataArray, 'eid');
                    $scopus_citation = $this->getMetadataValue($metadataArray, 'scopus_citation') ?: 0;
                    $scopus_author_id = $this->getMetadataValue($metadataArray, 'scopus_author_id');
                    $isbn = $this->getMetadataValue($metadataArray, 'isbn');
                    $issn = $this->getMetadataValue($metadataArray, 'issn');
                    $upurl = $this->getMetadataValue($metadataArray, 'upurl');
                    $sherpa_url = $this->getMetadataValue($metadataArray, 'shepa_url');
                    $unpaywall = $this->getMetadataValue($metadataArray, 'unpaywall');
                    $crossref_citation = $this->getMetadataValue($metadataArray, 'crossref_citation') ?: 0;

                    $publicationsData[] = [
                        'uuid' => $publication['dspace_object_id'],
                        'title' => $title,
                        'publication_type' => $publication_type,
                        'abstract' => $abstract,
                        'journal' => $journal,
                        'date_issued' => $date_issued,
                        'isbn' => $isbn,
                        'issn' => $issn,
                        'doi' => $doi,
                        'upurl' => $upurl,
                        'sherpa_url' => $sherpa_url,
                        'authors' => $authors,
                        'scopus_author_id' => $scopus_author_id,
                        'volume' => $volume,
                        'eid' => $eid,
                        'unpaywall' => $unpaywall,
                        'scopus_citation' => $scopus_citation,
                        'crossref_citation' => $crossref_citation,
                    ];
                }
                $totalCount = count($publicationsData);
            }
        }

        // Apply search filter
        $searchQuery = $request->input('search');
        if ($searchQuery) {
            $filteredData = [];
            foreach ($publicationsData as $publication) {
                // Case-insensitive search across multiple fields
                if (
                    stripos($publication['title'] ?? '', $searchQuery) !== false ||
                    stripos($publication['authors'] ?? '', $searchQuery) !== false ||
                    stripos($publication['journal'] ?? '', $searchQuery) !== false ||
                    stripos($publication['publication_type'] ?? '', $searchQuery) !== false ||
                    stripos($publication['scopus_citation'] ?? '', $searchQuery) !== false ||
                    stripos($publication['crossref_citation'] ?? '', $searchQuery) !== false ||
                    // stripos($publication['abstract'] ?? '', $searchQuery) !== false ||
                    stripos($publication['date_issued'] ?? '', $searchQuery) !== false ||
                    stripos($publication['doi'] ?? '', $searchQuery) !== false
                ) {
                    $filteredData[] = $publication;
                }
            }
            $publicationsData = $filteredData;
        }

        // Apply type filter
        $typeFilter = $request->input('type');
        if ($typeFilter && $typeFilter !== 'all') {
            $publicationsData = array_filter($publicationsData, function ($item) use ($typeFilter) {
                return isset($item['publication_type']) &&
                    strtolower($item['publication_type']) === strtolower($typeFilter);
            });
        }

        // Apply year range filter
        $fromYear = $request->input('from_year');
        $toYear = $request->input('to_year');

        if ($fromYear || $toYear) {
            $publicationsData = array_filter($publicationsData, function ($item) use ($fromYear, $toYear) {
                if (empty($item['date_issued'])) {
                    return false;
                }

                $dateString = $item['date_issued'];

                // Clean up unwanted characters
                $dateString = trim($dateString, "\"' ");

                // If multiple values like "2019, 2019-01-01", take the first one
                if (strpos($dateString, ',') !== false) {
                    $dateString = explode(',', $dateString)[0];
                }

                // Extract just the year (regex finds first 4-digit number)
                if (preg_match('/\d{4}/', $dateString, $matches)) {
                    $year = (int) $matches[0];
                } else {
                    return false; // skip if no valid year
                }

                // Apply filter
                if ($fromYear && $year < $fromYear) {
                    return false;
                }

                if ($toYear && $year > $toYear) {
                    return false;
                }

                return true;
            });
        }

        // if (!empty($dateIssued)) {
        //     // Clean input: remove quotes & trim spaces
        //     $cleanDate = trim(str_replace('"', '', $dateIssued));

        //     // If multiple dates separated by comma, take the last one
        //     if (strpos($cleanDate, ',') !== false) {
        //         $parts = explode(',', $cleanDate);
        //         $cleanDate = trim(end($parts));
        //     }

        //     // Handle Excel numeric dates like 43468 → 2023-12-01
        //     if (is_numeric($cleanDate)) {
        //         $unixDate = ($cleanDate - 25569) * 86400;
        //         $cleanDate = gmdate('Y-m-d', $unixDate);
        //     }

        //     try {
        //         $year = \Carbon\Carbon::parse($cleanDate)->format('Y');
        //         $years[] = (int) $year;
        //     } catch (\Exception $e) {
        //         // Invalid date → skip silently
        //     }
        // }


        // $fromDate = $request->input('from_date'); // 'Y-m-d'
        // $toDate = $request->input('to_date');     // 'Y-m-d'

        // if ($fromDate || $toDate) {
        //     $publicationsData = array_filter($publicationsData, function ($item) use ($fromDate, $toDate) {
        //         if (empty($item['date_issued'])) {
        //             return false;
        //         }

        //         $date = \Carbon\Carbon::parse($item['date_issued']);

        //         if ($fromDate && $date->lt(\Carbon\Carbon::parse($fromDate))) {
        //             return false;
        //         }

        //         if ($toDate && $date->gt(\Carbon\Carbon::parse($toDate))) {
        //             return false;
        //         }

        //         return true;
        //     });
        // }

        // Manual pagination
        $page = $request->input('page', 1);
        $perPage = $request->input('perPage', 10);
        $offset = ($page - 1) * $perPage;

        $publicationsData = array_values($publicationsData); // Re-index array
        $currentPageItems = array_slice($publicationsData, $offset, $perPage);

        $publications = new LengthAwarePaginator(
            $currentPageItems,
            count($publicationsData),
            $perPage,
            $page,
            [
                'path' => url()->current(), // <-- Fixes incorrect pagination base path
                'query' => $request->query()
            ]
        );

        // 'path' => $request->url()
        // $minYear = !empty($years) ? min($years) : null;
        // $maxYear = !empty($years) ? max($years) : null;

        // $minDate = !empty($dates) ? min($dates) : null;
        // $maxDate = !empty($dates) ? max($dates) : null;
        $minYear = !empty($years) ? min($years) : null;
        $maxYear = !empty($years) ? max($years) : null;

        return view('content.pages.publicationlist', compact(
            'publications',
            'totalCount',
            'minYear',
            'maxYear',
            'fromYear',
            'toYear',
            'searchQuery',
            'typeFilter'
        ));
    }


    // Get metadata 
    private function getMetadataValue($metadataArray, $fieldName)
    {
        foreach ($metadataArray as $item) {
            if ($item['field_name'] === $fieldName) {
                return $item['text_value'];
            }
        }
        return '';
    }

    // Create Json File
    public function createJsonFile()
    {

        $title =  $this->title;
        $publication_type = $this->publication_type;
        $journal = $this->journal;
        $isbn = $this->isbn;
        $issn = $this->issn;
        $date_issued = $this->date_issued;
        $sherpa_url = $this->sherpa_url;
        $unpaywall = $this->unpaywall;
        $upurl = $this->upurl;
        $crossref_citation = $this->crossref_citation;
        // $scopus_citation = $this->scopus_citation;
        $authors = $this->authors;
        $scopus_author_id = $this->scopus_author_id;
        $doi = $this->doi;
        $eid = $this->eid;
        $volume = $this->volume;
        $abstract = $this->abstract;

        // Step 1: Get dspace_object_id for "Publication" type
        $filteredObjects = DB::table('metadatavalue')
            ->select('dspace_object_id')
            ->where('metadata_field_id', 7)
            ->where('text_value', 'Publication')
            ->whereIn('dspace_object_id', function ($query) {
                $query->select('item_id')
                    ->from('public.collection2item')
                    ->whereIn('collection_id', function ($subQuery) {
                        $subQuery->select('uuid')->from('collection');
                    });
            })
            ->distinct()
            ->pluck('dspace_object_id',)
            ->toArray();

        // Step 1: Get main metadata (excluding scopus_citation)
        $metadataQuery = DB::table('metadatavalue as mv')
            ->select(
                'mv.dspace_object_id',
                DB::raw("
                            CASE 
                            WHEN mv.metadata_field_id = $title THEN 'title'
                            WHEN mv.metadata_field_id = $publication_type THEN 'publication_type'
                            WHEN mv.metadata_field_id = $date_issued THEN 'date_issued'
                            WHEN mv.metadata_field_id = $isbn THEN 'isbn'
                            WHEN mv.metadata_field_id = $issn THEN 'issn'
                            WHEN mv.metadata_field_id = $upurl THEN 'upurl'
                            WHEN mv.metadata_field_id = $authors THEN 'authors'
                            WHEN mv.metadata_field_id = $doi THEN 'doi'
                            WHEN mv.metadata_field_id = $eid THEN 'eid'
                            WHEN mv.metadata_field_id = $journal THEN 'journal'
                            WHEN mv.metadata_field_id = $scopus_author_id THEN 'scopus_author_id'
                            WHEN mv.metadata_field_id = $volume THEN 'volume'
                            WHEN mv.metadata_field_id = $unpaywall THEN 'unpaywall'
                            WHEN mv.metadata_field_id = $crossref_citation THEN 'crossref_citation'
                            WHEN mv.metadata_field_id = $sherpa_url THEN 'sherpa_url'
                            WHEN mv.metadata_field_id = $abstract THEN 'abstract'
                            ELSE 'unknown_field_' || mv.metadata_field_id
                            END AS field_name,
                            CASE    
                        WHEN mv.metadata_field_id IN ($authors, $scopus_author_id) THEN STRING_AGG(mv.text_value, ' || ') 
                        ELSE TRIM(BOTH '[]\"' FROM JSON_AGG(mv.text_value)::TEXT) 
                        END AS text_values
                        ")
            )
            ->whereIn('mv.dspace_object_id', $filteredObjects)
            ->whereIn('mv.metadata_field_id', [
                $title,
                $date_issued,
                $isbn,
                $issn,
                $upurl,
                $authors,
                $doi,
                $eid,
                $journal,
                $scopus_author_id,
                $volume,
                $unpaywall,
                $crossref_citation,
                $sherpa_url,
                $abstract,
                $publication_type
            ])
            ->groupBy('mv.dspace_object_id', 'mv.metadata_field_id');


        // Step 2: Get scopus_citation from cris_metrics in same structure
        $scopusCitationQuery = DB::table('cris_metrics as cm')
            ->select(
                'cm.resource_id as dspace_object_id',
                DB::raw("'scopus_citation' AS field_name"),
                DB::raw("CAST(cm.metriccount AS TEXT) AS text_values")
            )
            ->where('cm.metrictype', '=', 'scopusCitation')
            ->whereIn('cm.resource_id', $filteredObjects);


        // Step 3: Combine both using UNION ALL
        $combinedQuery = DB::table(DB::raw("({$metadataQuery->toSql()} UNION ALL {$scopusCitationQuery->toSql()}) as combined"))
            ->mergeBindings($metadataQuery)
            ->mergeBindings($scopusCitationQuery);


        // Step 4: Build final JSON structure
        $publications = DB::table(DB::raw("({$combinedQuery->toSql()}) as am"))
            ->mergeBindings($combinedQuery)
            ->select(
                'am.dspace_object_id',
                DB::raw("JSON_AGG(
                    JSON_BUILD_OBJECT(
                        'field_name', am.field_name,
                        'text_value', am.text_values
                        )
                        ) AS metadata_details")
            )
            ->groupBy('am.dspace_object_id')
            ->get();

        // Step 4: Store data in JSON file
        $jsonFilePath = storage_path('app/json/publicationlist.json');

        file_put_contents($jsonFilePath, json_encode($publications, JSON_PRETTY_PRINT));
    }


    // Delete function
    public function delete($uuid, DSpaceApiService $apiService)
    {
        if (session()->has('api_token')) {
            $authorizationHeader = session()->get('api_token');

            $response = $apiService->delete('core/items/' . $uuid, $authorizationHeader);


            if ($response->getStatusCode() === 204) {
                $this->createJsonFile();
                return redirect()->back()->with('success', 'Publication deleted successfully.');
            } else {
                return redirect()->back()->with('error', 'Failed to delete publication.');
            }
        } else {
            return redirect()->route('auth-login-basic');
        }
    }

    // Update list of publication
    public function publicationlistupdate(Request $request)
    {
        $this->createJsonFile();

        session()->flash('success', 'PublicationsList Updated Successfully!');

        return response()->json(['success' => session('success')]);
    }


    // display publication based on collection id
    public function displaypublications($id, $communityid)
    {
        $title = $this->title;
        // Run the query to get the publication IDs
        $filteredObjects = DB::select("
                                            SELECT DISTINCT mv_outer.dspace_object_id
                                            FROM metadatavalue AS mv_outer
                                            WHERE mv_outer.metadata_field_id = 7
                                            AND mv_outer.text_value = 'Publication'
                                            AND mv_outer.dspace_object_id IN (
                                                SELECT c2i.item_id
                                                FROM collection2item AS c2i
                                                JOIN metadatavalue AS mv ON c2i.collection_id = mv.dspace_object_id
                                                JOIN collection AS col ON mv.dspace_object_id = col.uuid
                                                WHERE mv.metadata_field_id = 7
                                                AND mv.text_value = 'Publication'
                                                AND c2i.collection_id = ?
                                            )
                                        ", [$id]);

        // Extract dspace_object_id values
        $publicationIds = array_column($filteredObjects, 'dspace_object_id');

        // Load JSON file
        // $filePath = public_path('json/publicationlist.json');
        $filePath = storage_path('app/json/publicationlist.json');
        $publicationsData = [];
        $years = [];

        if (file_exists($filePath)) {
            $fileContent = file_get_contents($filePath);
            $publicationsRaw = json_decode($fileContent, true);

            if (!empty($publicationsRaw)) {
                foreach ($publicationsRaw as $publication) {
                    // Only include publications that match our IDs
                    if (in_array($publication['dspace_object_id'], $publicationIds)) {
                        $metadataArray = json_decode($publication['metadata_details'], true);

                        // Year collection
                        // $dateIssued = $this->getMetadataValue($metadataArray, 'date_issued');
                        // if (!empty($dateIssued)) {
                        //     $year = \Carbon\Carbon::parse($dateIssued)->format('Y');
                        //     $years[] = (int) $year;
                        // }
                        // Year collection
                        $dateIssued = $this->getMetadataValue($metadataArray, 'date_issued');

                        if (!empty($dateIssued)) {

                            // Clean unexpected characters
                            $cleanDate = trim($dateIssued);
                            $cleanDate = str_replace(['"', '[', ']', "'"], '', $cleanDate);

                            // If multiple dates exist, take the first one
                            if (str_contains($cleanDate, ',')) {
                                $cleanDate = explode(',', $cleanDate)[0];
                            }

                            $year = \Carbon\Carbon::parse(trim($cleanDate))->format('Y');

                            $years[] = (int) $year;
                        }


                        $publicationsData[] = [
                            'uuid' => $publication['dspace_object_id'],
                            'title' => $this->getMetadataValue($metadataArray, 'title'),
                            'publication_type' => $this->getMetadataValue($metadataArray, 'publication_type'),
                            'abstract' => $this->getMetadataValue($metadataArray, 'abstract'),
                            'journal' => $this->getMetadataValue($metadataArray, 'journal'),
                            'date_issued' => $dateIssued,
                            'isbn' => $this->getMetadataValue($metadataArray, 'isbn'),
                            'issn' => $this->getMetadataValue($metadataArray, 'issn'),
                            'doi' => $this->getMetadataValue($metadataArray, 'doi'),
                            'upurl' => $this->getMetadataValue($metadataArray, 'upurl'),
                            'sherpa_url' => $this->getMetadataValue($metadataArray, 'shepa_url'),
                            'authors' => $this->getMetadataValue($metadataArray, 'authors'),
                            'scopus_author_id' => $this->getMetadataValue($metadataArray, 'scopus_author_id'),
                            'volume' => $this->getMetadataValue($metadataArray, 'volume'),
                            'eid' => $this->getMetadataValue($metadataArray, 'eid'),
                            'unpaywall' => $this->getMetadataValue($metadataArray, 'unpaywall'),
                            'scopus_citation' => $this->getMetadataValue($metadataArray, 'scopus_citation') ?: 0,
                            'crossref_citation' => $this->getMetadataValue($metadataArray, 'crossref_citation') ?: 0,
                        ];
                    }
                }
            }
        }

        // Get collection name
        $name = DB::table('metadatavalue')
            ->where('dspace_object_id', $id)
            ->where('metadata_field_id', $title)
            ->value('text_value');

        // Get request parameters for filtering
        $request = request();
        $searchQuery = $request->input('search');
        $typeFilter = $request->input('type');
        $fromYear = $request->input('from_year');
        $toYear = $request->input('to_year');
        $totalCount = count($publicationsData);

        // Apply search filter
        if ($searchQuery) {
            $filteredData = [];
            foreach ($publicationsData as $publication) {
                // Case-insensitive search across multiple fields
                if (
                    stripos($publication['title'] ?? '', $searchQuery) !== false ||
                    stripos($publication['authors'] ?? '', $searchQuery) !== false ||
                    stripos($publication['journal'] ?? '', $searchQuery) !== false ||
                    stripos($publication['publication_type'] ?? '', $searchQuery) !== false ||
                    stripos((string)$publication['scopus_citation'] ?? '', $searchQuery) !== false ||
                    stripos((string)$publication['crossref_citation'] ?? '', $searchQuery) !== false ||
                    stripos($publication['date_issued'] ?? '', $searchQuery) !== false ||
                    stripos($publication['doi'] ?? '', $searchQuery) !== false
                ) {
                    $filteredData[] = $publication;
                }
            }
            $publicationsData = $filteredData;
        }

        // Apply type filter
        if ($typeFilter && $typeFilter !== 'all') {
            $publicationsData = array_filter($publicationsData, function ($item) use ($typeFilter) {
                return isset($item['publication_type']) &&
                    strtolower($item['publication_type']) === strtolower($typeFilter);
            });
        }

        // Apply year range filter
        if ($fromYear || $toYear) {
            $publicationsData = array_filter($publicationsData, function ($item) use ($fromYear, $toYear) {
                if (empty($item['date_issued'])) {
                    return false;
                }

                $year = \Carbon\Carbon::parse($item['date_issued'])->format('Y');

                if ($fromYear && $year < $fromYear) {
                    return false;
                }

                if ($toYear && $year > $toYear) {
                    return false;
                }

                return true;
            });
        }

        // Manual pagination
        $page = $request->input('page', 1);
        $perPage = $request->input('perPage', 10);
        $offset = ($page - 1) * $perPage;

        $publicationsData = array_values($publicationsData); // Re-index array
        $currentPageItems = array_slice($publicationsData, $offset, $perPage);

        $publications = new \Illuminate\Pagination\LengthAwarePaginator(
            $currentPageItems,
            count($publicationsData),
            $perPage,
            $page,
            [
                'path' => url()->current(), // <-- Fixes incorrect pagination base path
                'query' => $request->query()
            ]
        );

        $minYear = !empty($years) ? min($years) : null;
        $maxYear = !empty($years) ? max($years) : null;

        return view('content.pages.publication', compact(
            'publications',  // This is now a paginator object like in the index function
            'totalCount',
            'minYear',
            'maxYear',
            'fromYear',
            'toYear',
            'searchQuery',
            'typeFilter',
            'name',
            'id',
            'communityid'
        ));
    }
}
