<?php

namespace App\Http\Controllers\pages;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use App\Services\DSpaceApiService;


class RegistriesController extends Controller
{
  public function index()
  {
    // Fetch data from the database
    $metadata = DB::table('metadataschemaregistry')->select('metadata_schema_id', 'namespace', 'short_id')->get();
    return view('content.pages.registries', compact('metadata'));
  }
  // addMetadataRegistry Function
  public function addMetadataRegistry(Request $request, DSpaceApiService $apiService)
  {
    // Validate the incoming request
    $request->validate([
      'name' => 'required|string|max:32|regex:/^[a-zA-Z0-9_-]+$/',
      'namespace' => 'required|url',
    ]);

    // Get the API token from session
    $authToken = session('api_token');
    if (!$authToken) {
      return back()->with('error', 'Authorization token not found.');
    }

    // Prepare the data payload
    $data = [
      'prefix' => $request->name,
      'namespace' => $request->namespace,
    ];

    try {
      // Send the request via the API service
      $response = $apiService->post('core/metadataschemas', $data, $authToken);

      if ($response->getStatusCode() === 201) {
        return back()->with('success', 'Metadata schema created successfully.');
      }

      return back()->with('error', 'Failed to create metadata schema: ' . $response->getBody());
    } catch (\Exception $e) {
      return back()->with('error', 'An error occurred: ' . $e->getMessage());
    }
  }

  // deleteMetadataRegistry
  public function deleteMetadataRegistry($id, DSpaceApiService $apiService)
  {
    $authorizationHeader = Session::get('api_token');

    $response = $apiService->delete('core/metadataschemas/' . $id, $authorizationHeader);

    if ($response->getStatusCode() === 204) {
      // Redirect to the desired view with a success message
      return redirect()->back()->with('success', 'metadata deleted successfully');
    } else {
      // Redirect back with an error message or handle the error accordingly
      return redirect()->back()->with('error', 'Failed to delete process');
    }
  }

  // updateMetadataRegistry
  public function updateMetadataRegistry(Request $request)
  {
    // Validate input fields
    $request->validate([
      'metadata_schema_id' => 'required|integer',
      'name' =>  'required|string|max:32|regex:/^[a-zA-Z0-9_-]+$/|unique:metadataschemaregistry,short_id,' . $request->metadata_schema_id . ',metadata_schema_id',
      'namespace' => 'required|url'
    ]);
    $updated = DB::table('metadataschemaregistry')
      ->where('metadata_schema_id', $request->metadata_schema_id)
      ->update([
        'namespace' => $request->namespace,
        'short_id' => $request->name, // name from the form = short_id in DB
      ]);


    if ($updated) {
      return redirect()->back()->with('success', 'Metadata schema updated successfully.');
    } else {
      return redirect()->back()->with('error', 'Failed to update metadata schema.');
    }
  }

  // RegistriesController.php
  // -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

  // showMetadataSchema
  public function showMetadataSchema($namespace)
  {
    $schemaId = DB::table('metadataschemaregistry')
      ->where('short_id', $namespace)
      ->value('metadata_schema_id');


    $metadata = DB::table('metadatafieldregistry as mfr')
      ->join('metadataschemaregistry as msr', 'mfr.metadata_schema_id', '=', 'msr.metadata_schema_id')
      ->where('msr.short_id', $namespace)
      ->select(
        'mfr.metadata_field_id',
        'mfr.element',                  // ✅ ADD THIS
        'mfr.qualifier',                // ✅ ADD THIS
        DB::raw("CONCAT(msr.short_id, '.', mfr.element, 
                     CASE 
                         WHEN mfr.qualifier IS NOT NULL THEN CONCAT('.', mfr.qualifier)
                         ELSE ''
                     END) as field_name"),
        'mfr.scope_note'
      )
      ->get();


    // Generate the full URL dynamically
    $fullNamespaceUrl = url("/schema/" . $namespace);
    return view('content.pages.schema-details', compact('metadata', 'namespace', 'fullNamespaceUrl', 'schemaId'));
  }

  // addMetadataSchema
  public function addMetadataSchema(Request $request, DSpaceApiService $apiService)
  {
    // Get API token
    $authToken = session('api_token');
    if (!$authToken) {
      return back()->with('error', 'Authorization token not found.');
    }

    // Construct API endpoint
    $schemaId = $request->input('schemaId');
    $endpoint = "core/metadatafields?schemaId=" . $schemaId;

    // Prepare data
    $data = [
      'element'   => $request->element,
      'qualifier' => $request->qualifier,
      'scopeNote' => $request->scopeNote,
    ];

    try {
      // Call API
      $response = $apiService->post($endpoint, $data, $authToken);

      if ($response->getStatusCode() === 201) {
        return back()->with('success', 'Metadata field created successfully.');
      }

      return back()->with('error', 'Failed to create metadata field: ' . $response->getBody());
    } catch (\Exception $e) {
      return back()->with('error', 'An error occurred: ' . $e->getMessage());
    }
  }

  // deleteMetadataSchema
  public function deleteMetadataSchema($id, DSpaceApiService $apiService)
  {

    $authorizationHeader = session()->get('api_token');
    
    $response = $apiService->delete('core/metadatafields/' . $id, $authorizationHeader);

    if ($response->getStatusCode() === 204) {
      return redirect()->back()->with('success', 'Metadata field deleted successfully.');
    } else {
      return redirect()->back()->with('error', 'Failed to delete metadata field.');
    }
  }

  // updateMetadataSchema
  public function updateMetadataSchema(Request $request)
  {
    $request->validate([
      'metadata_field_id' => 'required|integer',
      'element' => 'required|string',
      'qualifier' => 'nullable|string',
      'scope_note' => 'nullable|string',
    ]);

    DB::table('metadatafieldregistry')
      ->where('metadata_field_id', $request->metadata_field_id)
      ->update([
        'element' => $request->element,
        'qualifier' => $request->qualifier,
        'scope_note' => $request->scope_note,
      ]);

    return redirect()->back()->with('success', 'Metadata field updated successfully!');
  }
}
